<?php

class Crossroads_Collector_AddressesController extends Mage_Core_Controller_Front_Action {
    /**
     * @api {get} /Crossroads_Collector/addresses
     * @apiName Crossroads_Collector_getAddresses
     * @apiGroup Collector
     * @apiDescription Returns an array of addresses. If used with Crossroads_API, do not use this endpoint,
     * instead use the endpoint available on /payment/Crossroads_Collector/addresses.
     *
     * @apiParam {string} invId  The inhabitant id for the buyer
     *
     * @apiSuccess {String}   prefix
     * @apiSuccess {String}   firstname
     * @apiSuccess {String}   middlename
     * @apiSuccess {String}   lastname
     * @apiSuccess {String}   suffix
     * @apiSuccess {String}   company
     * @apiSuccess {String[]} street
     * @apiSuccess {String}   postcode
     * @apiSuccess {String}   city
     * @apiSuccess {String}   regionId
     * @apiSuccess {String}   countryId (SE, NO, DK)
     * @apiSuccess {String}   telephone
     * @apiSuccess {String}   fax
     */
    public function indexAction() {
        if($this->getRequest()->getMethod() !== "GET") {
            return $this->getResponse()
                ->setHttpResponseCode(405);
        }

        $id = $this->getRequest()->getParam("invId");

        if( ! $id) {
            return $this->getResponse()
                ->setHttpResponseCode(400)
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([ "message" => "Missing query parameter invId" ]));
        }

        $store  = Mage::app()->getStore();
        $locale = Mage::app()->getLocale()->getLocaleCode();

        return $this->runTry(function() use($store, $locale, $id) {
            return $this->getResponse()
                ->setHttpResponseCode(200)
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode(Mage::helper("Crossroads_Collector/soap")->fetchAddresses($store, $locale, $id)));
        });
    }

    public function saveSSNAction() {
        $ssn = $this->getRequest()->getParam("ssn");
        Mage::getSingleton('core/session')->setCollectorSsn($ssn);
    }

    public function setAction() {
        $ssn       = $this->getRequest()->getParam("invId");
        $addressId = intval($this->getRequest()->getParam("id"));

        $store  = Mage::app()->getStore();
        $locale = Mage::app()->getLocale()->getLocaleCode();

        return $this->runTry(function() use($store, $locale, $ssn, $addressId) {
            $addresses = Mage::helper("Crossroads_Collector/soap")->fetchAddresses($store, $locale, $ssn);

            if (empty($addresses[$addressId])) {
                return $this->getResponse()
                    ->setHttpResponseCode(500)
                    ->setHeader("Content-Type", "application/json", true)
                    ->setBody(json_encode([ "message" => "Address with index $addressId does not exist" ]));
            }

            $address = $addresses[$addressId];

            $billingAddress = Mage::getSingleton('checkout/session')->getQuote()->getBillingAddress();
            $this->updateAddress($billingAddress, $address);

            Mage::getSingleton('core/session')->setCollectorSsn($ssn);

            return $this->getResponse()
                ->setHttpResponseCode(200);
        });
    }

    /**
     * Update a Magento address with post values and save it
     *
     * @param object $address The Magento address
     * @param array  $post    Array holding the post values
     *
     * @return void
     */
    protected function updateAddress($address, $post)
    {
        if (array_key_exists("street", $post)) {
            $address->setStreet(trim(implode("\n", $post["street"])));
        }

        if (array_key_exists("firstname", $post)) {
            $address->setFirstname($post["firstname"]);
        }

        if (array_key_exists("lastname", $post)) {
            $address->setLastname($post["lastname"]);
        }

        if (array_key_exists("postcode", $post)) {
            $address->setPostcode($post["postcode"]);
        }

        if (array_key_exists("city", $post)) {
            $address->setCity($post["city"]);
        }

        if (array_key_exists("phone_number", $post)) {
            $address->setTelephone($post["phone_number"]);
        }

        $address->setCompany($post["company"]);

        $address->save();
    }

    protected function runTry($callback) {
        try {
            return $callback();
        }
        catch(Crossroads_Collector_Helper_Soap_Exception $e) {
            list($status, $msg, $code) = Crossroads_Collector_Helper_Soap_Exception::faultcodeToResponse($e->getFaultcode());

            // Crossroads_API_ResponseExceptions should not be logged, they are responses
            // to be sent to the client
            return $this->getResponse()
                ->setHttpResponseCode($status)
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode([
                "message"   => $this->__($code),
                "errorCode" => $code,
                "data"      => null
            ]));
        }
        catch(Exception $e) {
            // Copied from Crossroads_API_Controller_Super from Crossroads_API module
            $this->getResponse()->setHttpResponseCode(500);

            Mage::logException($e);

            if(Mage::getIsDeveloperMode()) {
                return $this->getResponse()
                    ->setHeader("Content-Type", "application/json", true)
                    ->setBody(json_encode([
                        "message"   => $e->getMessage(),
                        "errorCode" => null,
                        "data"      => [
                            "code"  => $e->getCode(),
                            "file"  => $e->getFile(),
                            "line"  => $e->getLine(),
                            "trace" => array_map(function($l) {
                                if(array_key_exists("class", $l)) {
                                    return [
                                        "call" => $l["class"].$l["type"].$l["function"],
                                        "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                        "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                    ];
                                }

                                return [
                                    "call" => $l["function"],
                                    "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                    "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                ];
                            }, $e->getTrace())
                        ]
                    ]));
            }
        }
    }
}
