<?php

/**
 * Wrapping exception for SoapFault used when Crossroads_API_ResponseException is not available.
 */
class Crossroads_Collector_Helper_Soap_Exception extends Crossroads_Collector_Exception {
    protected $faultcode;

    protected $faultstring;

    /**
     * Converts a SOAP faultcode from Collector into a [HTTP response code, message, error code] tuple.
     *
     * @param  string
     * @return array
     */
    public static function faultcodeToResponse($faultcode) {
        switch($faultcode) {
            case Crossroads_Collector_Helper_Soap::ERROR_INDIVIDUAL_NOT_SUPPORTED:
                return [403, "Collector Bank store id not set for primary individual payment.", 12010];
            case "s:INVALID_REGISTRATION_NUMBER":
                return [400, "Invalid registration number", 12001];
            case "s:DENIED_TO_PURCHASE":
                return [403, "Denied purchase", 12002];
            case "s:CREDIT_CHECK_DENIED":
                return [403, "Credit check denied", 12003];
            case "s:ADDRESS_NOT_FOUND":
                return [400, "Address not found", 12004];
            case "s:RESERVATION_NOT_APPROVED":
                return [403, "Reservation not approved", 12005];
            case "s:PURCHASE_AMOUNT_GREATER_THAN_MAX_CREDIT_AMOUNT":
                return [403, "Purchase amount too large", 12006];
            case "s:AGREEMENT_RULES_VALIDATION_FAILED":
                // Address does not match expected address for that RegNo
                return [403, "Agreement rules validation failed, supplied information does not match", 12007];
            case "s:AUTHORIZATION_FAILED":
                return [500, "Collector API call authorization failed", 12012];
            case "s:UNHANDLED_EXCEPTION":
                return [500, "Collector API call failed", 12011];
            case "s:VALIDATION_ADDRESS2_LENGTH":
                return [400, "Address2 too long", 12015];
            default:
                return [500, sprintf("Unknown SoapFault: '%s'", $faultcode), 0];
        }
    }

    public static function create(SoapFault $e) {
        list($_status, $_msg, $code) = self::faultcodeToResponse($e->faultcode);

        $newE = new Crossroads_Collector_Helper_Soap_Exception(sprintf("SoapFault: %s (%s): %s", $e->getMessage(), $e->faultcode, $e->faultstring), $code, $e);

        $newE->setFaultcode($e->faultcode);
        $newE->setFaultstring($e->faultstring);

        return $newE;
    }

    public function setFaultcode($code) {
        $this->faultcode = $code;
    }

    public function getFaultcode() {
        return $this->faultcode;
    }

    public function setFaultstring($string) {
        $this->faultstring = $string;
    }

    public function getFaultstring() {
        return $this->faultstring;
    }

    public function asCrossroadsApiException() {
        if($this->faultcode === "s:UNHANDLED_EXCEPTION") {
            return $this;
        }

        list($status, $msg, $code) = self::faultcodeToResponse($this->faultcode);

        return Crossroads_API_ResponseException::create($status, $msg, null, $code);
    }
}
