<?php

class Crossroads_Collector_Helper_Soap {
    /**
     * Event triggered in reserveAmount after all items have been added to the Klarna object but before
     * shipping has been added.
     *
     * Params:
     *  * order:      Mage_Sales_Model_Order
     *  * order_rows: ArrayObject
     *  * is_company: True if the order is placed by a company
     */
    const EVENT_RESERVE_AMOUNT_POST_ADD_ARTICLES = "crossroads_collector_reserve_amount_post_add_articles";

    const SOAP_NAMESPACE = "http://schemas.ecommerce.collector.se/v30/InvoiceService";

    const ERROR_INDIVIDUAL_NOT_SUPPORTED = "CROSSROADS_ERROR_INDIVIDUAL_NOT_SUPPORTED";

    /**
     * Splits a locale in the format sv_SE into [sv, SE]
     *
     * @param  string
     * @return Array<string>
     */
    protected static function splitLocale($locale) {
        return explode("_", trim(strtr($locale, "-:", "__"), " \t\r\n_-"));
    }

    /**
     * Obtains the country code from a locale.
     *
     * @param  string
     * @return string
     */
    public static function localeToCountry($locale) {
        $parts = self::splitLocale($locale);

        return end($parts);
    }

    /**
     * Obtains the language code from a locale.
     *
     * @param  string
     * @return string
     */
    public static function localeToLanguage($locale) {
        $parts = self::splitLocale($locale);

        return reset($parts);
    }

    protected function formatSoapFault($e) {
        if(Crossroads_Collector_Helper_Soap_RecoverableException::isRecoverableFaultcode($e->faultcode)) {
            return Crossroads_Collector_Helper_Soap_RecoverableException::create($e);
        }

        return Crossroads_Collector_Helper_Soap_Exception::create($e);
    }

    protected function getWsdlUrl($test) {
        return $test ? "https://ecommercetest.collector.se/v3.0/InvoiceServiceV33.svc?wsdl" : "https://ecommerce.collector.se/v3.0/InvoiceServiceV33.svc?wsdl";
    }

    public function faultcodeToResponse($faultcode) {
        return Crossroads_Collector_Helper_Soap_Exception::faultcodeToResponse($faultcode);
    }

    protected function createClient($store, $ip) {
        $test     = $store->getConfig("payment/Crossroads_Collector/test");
        $username = $store->getConfig("payment/Crossroads_Collector/username");
        $password = $store->getConfig("payment/Crossroads_Collector/password");

        $client = new SoapClient($this->getWsdlUrl($test), [
            "login"      => $username,
            "password"   => $password,
            "cache_wsdl" => WSDL_CACHE_BOTH,
            "trace"      => Mage::getIsDeveloperMode()
        ]);

        $client->__setSoapHeaders([
            new SoapHeader(self::SOAP_NAMESPACE, "Username", $username),
            new SoapHeader(self::SOAP_NAMESPACE, "Password", $password),
            new SoapHeader(self::SOAP_NAMESPACE, "ClientIpAddress", $ip),
        ]);

        return $client;
    }

    public function fetchAddresses($store, $locale, $invId) {
        $helper  = Mage::helper("Crossroads_Collector");
        $ip      = Mage::helper("core/http")->getRemoteAddr();
        $client  = $this->createClient($store, $ip);
        // We always use the primary store here to obtain the addresses
        $storeId = $store->getConfig($helper->getStoreIdKey(false, false));

        if( ! $storeId) {
            list($status, $msg, $code) = Crossroads_Collector_Helper_Soap_Exception::faultcodeToResponse(self::ERROR_INDIVIDUAL_NOT_SUPPORTED);

            if(Mage::helper("core")->isModuleEnabled("Crossroads_API")) {
                throw Crossroads_API_ResponseException::create($status, $msg, null, $code);
            }

            $e = new Crossroads_Collector_Helper_Soap_Exception();

            $e->setFaultcode(self::ERROR_INDIVIDUAL_NOT_SUPPORTED);
            $e->setFaultstring($msg);

            throw $e;
        }

        try {
            $data = $client->GetAddress([
                "RegNo"           => $invId,
                "CountryCode"     => self::localeToCountry($locale),
                "StoreId"         => $storeId,
            ]);
        }
        catch(SoapFault $e) {
            if(Mage::getIsDeveloperMode()) {
                Mage::log(sprintf("Crossroads_Collector: SOAP::GetAddress request failed:\nRequest: %s\nResponse: %s", $client->__getLastRequest(), $client->__getLastResponse()));
            }

            throw $this->formatSoapFault($e);
        }

        return array_values(array_map(function($a) use($data) {
            return [
                "prefix"     => null,
                "firstname"  => $data->Firstname,
                "middlename" => null,
                "lastname"   => $data->Lastname,
                "suffix"     => null,
                "company"    => null,
                "street"     => array_filter([$a->Address1, $a->Address2]),
                "postcode"   => $a->PostalCode,
                "city"       => $a->City,
                "regionId"   => null,
                "countryId"  => $a->CountryCode,
                "telephone"  => null,
                "fax"        => null,
            ];
        }, array_filter(get_object_vars($data->Addresses)) ?: []));
    }

    public function captureInvoice($store, $storeId, $order) {
        $invoiceNo = $order->getPayment()->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);
        $client    = $this->createClient($store, $order->getRemoteIp());

        try {
            return $client->ActivateInvoice([
                "StoreId"      => $storeId,
                "CountryCode"  => $order->getBillingAddress()->getCountryId(),
                "InvoiceNo"    => $invoiceNo
            ]);
        }
        catch(SoapFault $e) {
            if(Mage::getIsDeveloperMode()) {
                Mage::log(sprintf("Crossroads_Collector: SOAP::ActivateInvoice request failed:\nRequest: %s\nResponse: %s", $client->__getLastRequest(), $client->__getLastResponse()));
            }

            throw $this->formatSoapFault($e);
        }
    }

    public function cancelInvoice($store, $storeId, $order) {
        $invoiceNo = $order->getPayment()->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);
        $client    = $this->createClient($store, $order->getRemoteIp());

        try {
            return $client->CancelInvoice([
                "StoreId"      => $storeId,
                "CountryCode"  => $order->getBillingAddress()->getCountryId(),
                "InvoiceNo"    => $invoiceNo
            ]);
        }
        catch(SoapFault $e) {
            if(Mage::getIsDeveloperMode()) {
                Mage::log(sprintf("Crossroads_Collector: SOAP::CancelInvoice request failed:\nRequest: %s\nResponse: %s", $client->__getLastRequest(), $client->__getLastResponse()));
            }

            throw $this->formatSoapFault($e);
        }
    }

    protected function orderIsVirtual($order) {
        // Older Magento versions do not have the isVirtual method
        if(method_exists($order, "isVirtual")) {
            return $order->isVirtual();
        }

        return $order->getIsVirtual();
    }

    /**
     * Creates an invoice with Collector.
     *
     * @param  Mage_Core_Model_Store
     * @param  integer  Collector store id
     * @return object   Returns object with InvoiceNo, InvoiceStatus
     */
    public function createInvoice($store, $storeId, $order) {
        $helper    = Mage::helper("Crossroads_Collector");
        $client    = $this->createClient($store, $order->getRemoteIp());
        $ssn       = $order->getPayment()->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_SSN);
        $items     = $order->getAllVisibleItems();
        $isCompany = $order->getPayment()->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);

        // TODO: if it is a person it is required to be the same address for both shipping and billing

        if( ! $ssn) {
            // Field is required, and enforced by the observer in Crossroads_Klarna_Helper_Data::checkoutPostVerifyQuote
            // SSN is also required for business customers, not only customers
            throw new Exception("Crossroads_Collector: Expected required additional information '".Crossroads_Collector_Helper_Data::FIELD_SSN."' not present in order payment data. This should not happen if payment is triggered properly.");
        }

        $orderRows = new ArrayObject();

        foreach($items as $item) {
            $rowTotal = $item->getBaseRowTotal()
                  + $item->getBaseTaxAmount()
                  + $item->getBaseHiddenTaxAmount()
                  - $item->getBaseDiscountAmount();

            $orderRows->append([
                // Article number, string
                "ArticleId"   => $item->getSku(),
                // Article name/title, string
                "Description" => substr($item->getName(), 0, 50),
                // Quantity, int
                "Quantity"    => (double)$item->getQtyOrdered(),
                // Price per item, float
                "UnitPrice"   => round($rowTotal / $item->getQtyOrdered(), 2),
                // Only whole percentage-units
                "VAT"         => ($item->getBaseRowTotal() - $item->getBaseDiscountAmount()) > 0 ?
                                 round(($item->getBaseTaxAmount() + $item->getBaseHiddenTaxAmount()) / ($item->getBaseRowTotal() - $item->getBaseDiscountAmount()) * 100, 0) :
                                 0,
            ]);
        }

        Mage::dispatchEvent(self::EVENT_RESERVE_AMOUNT_POST_ADD_ARTICLES, [
            "order"      => $order,
            "order_rows" => $orderRows,
            "is_company" => $isCompany,
        ]);

        if($order->getBaseShippingAmount() > 0) {
            $orderRows->append([
                "ArticleId"   => $order->getShippingMethod(),
                "Description" => substr($order->getShippingDescription(), 0, 50),
                "Quantity"    => 1,
                "UnitPrice"   => round($order->getBaseShippingInclTax(), 2),
                // Only whole percentage-units
                "VAT"         => round($order->getBaseShippingTaxAmount() / $order->getBaseShippingAmount() * 100, 0)
            ]);
        }

        $deliveryAddress = $this->orderIsVirtual($order) ? $order->getBillingAddress() : $order->getShippingAddress();

        try {
            return $client->AddInvoice([
                "StoreId"               => $storeId,
                "CountryCode"           => $order->getBillingAddress()->getCountryId(),
                "RegNo"                 => $ssn,
                "Currency"              => $order->getBaseCurrencyCode(),
                "CustomerNo"            => $order->getCustomerId(),
                "OrderNo"               => $order->getIncrementId(),
                "OrderDate"             => gmdate("Y-m-d\TH:i:s\Z"),
                "InvoiceRows"           => $orderRows->getArrayCopy(),
                // Per-order invoice (part payment (2, 4, 5) cannot be used with company):
                "InvoiceType"           => 0,
                "InvoiceAddress"        => $this->prepareAddress($order->getBillingAddress(), $order->getCustomerEmail(), $isCompany),
                "DeliveryAddress"       => $this->prepareAddress($deliveryAddress, $order->getCustomerEmail(), $isCompany),
                // We cannot use this option for orders which are not immediately deliverable, so we never activate
                // immediately
                "ActivationOption"      => 0,
                "Reference"             => $isCompany ? sprintf("%s %s", $order->getBillingAddress()->getFirstname(), $order->getBillingAddress()->getLastname()) : null,
                // Not required in sweden
                "Gender"                => null,
                // TODO: FIXME
                "InvoiceDeliveryMethod" => (int)$store->getConfig("payment/Crossroads_Collector/invoice_delivery_method"),
                "PurchaseType"          => 0,
            ]);
        }
        catch(SoapFault $e) {
            if(Mage::getIsDeveloperMode()) {
                Mage::log(sprintf("Crossroads_Collector: SOAP::AddInvoice request failed:\nRequest: %s\nResponse: %s", $client->__getLastRequest(), $client->__getLastResponse()));
            }

            throw $this->formatSoapFault($e);
        }
    }

    protected function prepareAddress($address, $email, $isCompany) {
        $street = is_array($address->getStreet()) ? $address->getStreet() : array_map("trim", explode("\n", $address->getStreet()));

        return [
            "Firstname"       => $isCompany ? null : $address->getFirstname(),
            "Lastname"        => $isCompany ? null : $address->getLastname(),
            "CompanyName"     => $address->getCompany(),
            "Address1"        => array_key_exists(0, $street) ? $street[0] : null,
            "Address2"        => array_key_exists(1, $street) ? $street[1] : null,
            // "COAddress"    => null
            "PostalCode"      => $address->getPostcode(),
            "City"            => $address->getCity(),
            // "PhoneNumber"  => null
            "CellPhoneNumber" => $address->getTelephone(), // cellno
            "Email"           => $email,
            "CountryCode"     => $address->getCountryId(),
        ];
    }
}
