<?php

class Crossroads_Collector_Model_Collector extends Mage_Payment_Model_Method_Abstract
{
    protected $_code = "Crossroads_Collector";

    // Properties from mage model
    protected $_isGateway               = true;
    // Required to provide the ability to call the API, without this capture() would not be called
    protected $_isOnline                = true;
    // We do not perform separate reservations
    protected $_canUseForMultishipping  = false;
    protected $_canUseInternal          = true;
    protected $_canAuthorize            = true;
    // We do not allow refunds in Magento, this process is performed outside of Magento
    protected $_canRefund               = false;
    // We need to be able to review payment since we send it to Collector Bank for review.
    protected $_canReviewPayment        = true;
    protected $_canCapture              = true;
    protected $_canVoid                 = true;
    protected $_canCancel               = true;
    protected $_canFetchTransactionInfo = true;

    protected $_infoBlockType = "Crossroads_Collector/info_invoice";
    protected $_formBlockType = "Crossroads_Collector/form";

    /**
     * Check whether payment method can be used
     *
     * @param  Mage_Sales_Model_Quote|null
     * @return bool
     */
    public function isAvailable($quote = null) {
        $minimumOrderTotal = Mage::getStoreConfig(Crossroads_Collector_Helper_Data::CONFIG_PATH_MIN_ORDER_TOTAL, $quote ? $quote->getStoreId() : null);

        return (!$quote || $quote->getBaseGrandTotal() >= $minimumOrderTotal) &&
            parent::isAvailable($quote);
    }

    /**
     * Called through `Mage_Sales_Model_Quote_Payment::importData` to set data for the payment method.
     *
     * @param  Varien_Object
     * @return Mage_Payment_Model_Info
     */
    public function assignData($data) {
        $helper = Mage::helper("Crossroads_Collector");
        $info   = $this->getInfoInstance();

        $info->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_SSN, $data->getData("socialSecurityNumber"));

        $isCompany = $data->hasData("socialSecurityNumber") &&
            $helper->isOrganizationNumber($data->getData("socialSecurityNumber")) ||
            $data->hasData("isCompany") && $data->getData("isCompany");

        $info->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY, $isCompany);

        return parent::assignData($data);
    }

    /**
     * Updates the payment status given an invoice number and status code.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  string
     * @param  integer
     */
    public function updatePaymentStatus($payment, $invoiceNo, $invoiceStatusCode) {
        $order = $payment->getOrder();
        $store = Mage::app()->getStore($order->getStoreId());

        $payment->setTransactionId($invoiceNo);
        $payment->setIsTransactionClosed(false);
        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS, $invoiceStatusCode);

        switch($invoiceStatusCode) {
        case Crossroads_Collector_Helper_Data::INVOICE_STATUS_ON_HOLD:
            $payment->setIsTransactionPending(true);
            // Since we do not actually close the authorization transaction here, the order will be
            // pending payment review
            break;
        case Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY:
            $payment->setIsTransactionApproved(true);

            // Sometimes we approve the transaction immediately
            if($order->getState() === Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                $payment->accept();
            }

            break;
        case Crossroads_Collector_Helper_Data::INVOICE_STATUS_REJECTED:
            $this->fallbackCreateInvoice($payment);

            break;
        default:
            throw new Exception(sprintf("Crossroads_Collector: Unknown invoice status code '%s' for invoice %s and order %s", $invoiceStatusCode, $invoiceNo, $payment->getOrder()->getIncrementId()));
        }
    }

    protected function fallbackCreateInvoice($payment) {
        $helper    = Mage::helper("Crossroads_Collector");
        $soap      = Mage::helper("Crossroads_Collector/soap");
        $order     = $payment->getOrder();
        $store     = Mage::app()->getStore($order->getStoreId());
        $isCompany = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);
        $storeId   = $store->getConfig($helper->getStoreIdKey(true, $isCompany));

        $payment->setIsTransactionApproved(false);
        $payment->setIsTransactionPending(0);
        $payment->setIsTransactionClosed(1);

        // If we are using a fallback which doesn't exist we just cancel the order since it didn't work
        if($payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK) ||
            ! $store->getConfig("payment/Crossroads_Collector/use_fallback") ||
            ! $storeId) {
            Mage::log(sprintf("Crossroads_Collector: Fallback for order %s, company: %s", $order->getIncrementId(), $isCompany ? "Yes" : "No"));

            $order->cancel();
            $order->save();

            return;
        }

        $payment->addTransaction(Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH, null, false, Mage::helper("Crossroads_Collector")->__("Failed to authorize transaction %s, attempting fallback.", $payment->getTransactionId()));

        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK, true);
        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_STORE_ID, $storeId);

        try {
            Mage::log(sprintf("Crossroads_Collector: Attempting fallback for order %s in store %s", $order->getIncrementId(), $storeId));

            $result = $soap->createInvoice($store, $storeId, $order);

            Mage::log(sprintf("Crossroads_Collector: Created invoice with collector for order %s in store %s, got invoice number %s and status %s, using fallback.", $order->getIncrementId(), $storeId, $result->InvoiceNo, $result->InvoiceStatus));

            $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID, $result->InvoiceNo);
            $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS, $result->InvoiceStatus);

            $payment->setParentTransactionId($invoiceNo);
            $payment->setTransactionId($result->InvoiceNo);
            $payment->setIsTransactionClosed(false);

            switch($result->InvoiceStatus) {
            case Crossroads_Collector_Helper_Data::INVOICE_STATUS_ON_HOLD:
                $payment->setIsTransactionPending(true);
                // Since we do not actually close the authorization transaction here, the order will be
                // pending payment review
                break;
            case Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY:
                $payment->setIsTransactionApproved(true);

                // Sometimes we approve the transaction immediately
                if($order->getState() === Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    $payment->accept();
                }

                break;
            case Crossroads_Collector_Helper_Data::INVOICE_STATUS_REJECTED:
                $order->cancel();
                $order->save();
                break;
            }
        }
        catch(Exception $e) {
            Mage::logException($e);

            $order->cancel();
            $order->save();

            throw $e;
        }
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function authorize(Varien_Object $payment, $amount) {
        if( ! $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_SSN)) {
            throw new Crossroads_Collector_Exception("Crossroads_Collector: Missing required additional information ".Crossroads_Collector_Helper_Data::FIELD_SSN.".", 12009);
        }

        $ssn = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_SSN);

        $fallback  = false;
        $helper    = Mage::helper("Crossroads_Collector");
        $soap      = Mage::helper("Crossroads_Collector/soap");
        $order     = $payment->getOrder();
        $store     = Mage::app()->getStore($order->getStoreId());
        $isCompany = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);
        $storeId   = $store->getConfig($helper->getStoreIdKey($fallback, $isCompany));

        if( ! $storeId) {
            $e = new Exception(sprintf("Collector Bank store id not set for %s %s payment.", $fallback ? "fallback" : "primary", $isCompany ? "company" : "individual"));

            if(Mage::helper("core")->isModuleEnabled("Crossroads_API")) {
                throw Crossroads_API_ResponseException::create(400, sprintf("Collector payment is not enabled for %s.", $isCompany ? "company" : "individual"), null, $isCompany ? 12014 : 12013);
            }
            else {
                throw $e;
            }
        }

        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK, $fallback);
        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_STORE_ID, $storeId);

        try {
            try {
                $result = $soap->createInvoice($store, $storeId, $order);

                Mage::log(sprintf("Crossroads_Collector: Created invoice with collector for order %s in store %s, got invoice number %s and status %s.", $order->getIncrementId(), $storeId, $result->InvoiceNo, $result->InvoiceStatus));

                $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID, $result->InvoiceNo);

                $this->updatePaymentStatus($payment, $result->InvoiceNo, $result->InvoiceStatus);
            }
            catch(Crossroads_Collector_Helper_Soap_RecoverableException $e) {
                // Try the fallback if we have one
                Mage::log(sprintf("Crossroads_Collector: Got RecoverableException, attempting to use fallback for order %s", $order->getIncrementId()));

                $this->fallbackCreateInvoice($payment);
            }
        }
        catch(Exception $e) {
            // Common catch
            Mage::logException($e);

            $order->cancel();
            $order->save();

            if($e instanceof Crossroads_Collector_Helper_Soap_Exception && Mage::helper("core")->isModuleEnabled("Crossroads_API")) {
                throw $e->asCrossroadsApiException();
            }

            throw $e;
        }

        $payment->save();

        return $this;
    }

    /**
     * Captures amount from Collector from a previous reservation made using authorize.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function capture(Varien_Object $payment, $amount) {
        $auth = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);

        if( ! $auth || ! $auth->getTxnId()) {
            Mage::throwException("Crossroads_Collector: Failed to obtain authorization id for order '{$payment->getOrder()->getIncrementId()}'.");
        }

        $helper        = Mage::helper("Crossroads_Collector");
        $soap          = Mage::helper("Crossroads_Collector/soap");
        $order         = $payment->getOrder();
        $authTxnId     = $auth->getTxnId();
        $invoiceId     = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);
        $invoiceStatus = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS);
        $isFallback    = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK);
        $isCompany     = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);
        $store         = Mage::app()->getStore($order->getStoreId());
        $storeId       = $store->getConfig($helper->getStoreIdKey($isFallback, $isCompany));

        if($invoiceStatus != Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY) {
            // We can only activate preliminary invoices (the other statuses are either waiting on Collector Bank,
            // activated, or rejected).
            throw new Exception(sprintf("Crossroads_Collector: Invoice status for order '%s' is not PRELIMINARY (1) instead it is '%s', cannot capture payment of invoice '%s'.", $order->getIncrementId(), $invoiceStatus, $invoiceId));
        }

        try {
            $result = $soap->captureInvoice($store, $storeId, $order);

            Mage::log(sprintf("Crossroads_Collector: Captured invoice with collector for order %s in store %s, got payment reference %s", $order->getIncrementId(), $storeId, $result->PaymentReference));

            // If we have a response, it is successful and the invoice has been activated
            $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS, 2);
            $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_PAYMENT_REFERENCE, $result->PaymentReference);

        }
        catch(Exception $e) {
            Mage::logException($e);

            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage()." (#".$e->getCode().")", $e->getCode(), $e);
        }

        $payment->setParentTransactionId($invoiceId);
        $payment->setTransactionId($invoiceId);
        $payment->setIsTransactionPending(0);
        $payment->setIsTransactionClosed(1);

        return $this;
    }

    /**
     * Attempts to cancel a reservation with Collector, cancels the order even if we get an error.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function cancel(Varien_Object $payment) {
        try {
            return $this->void($payment);
        }
        catch(Exception $e) {
            Mage::logException($e);

            Mage::getSingleton("core/session")->addError($e->getMessage()." (#".$e->getCode().")");

            return $this;
        }
    }

    /**
     * Voids a reservation with Collector.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function void(Varien_Object $payment) {
        $auth = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);

        if( ! $auth || ! $auth->getTxnId()) {
            Mage::throwException("Crossroads_Collector: Failed to obtain authorization id for order '{$payment->getOrder()->getIncrementId()}'.");
        }

        $helper        = Mage::helper("Crossroads_Collector");
        $soap          = Mage::helper("Crossroads_Collector/soap");
        $order         = $payment->getOrder();
        $authTxnId     = $auth->getTxnId();
        $invoiceId     = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);
        $invoiceStatus = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS);
        $isFallback    = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK);
        $isCompany     = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);
        $store         = Mage::app()->getStore($order->getStoreId());
        $storeId       = $store->getConfig($helper->getStoreIdKey($isFallback, $isCompany));

        if( ! $invoiceId) {
            throw new Exception(sprintf("Crossroads_Collector: Missing required additional information '%s' in order '%s'.", Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID, $order->getIncrementId()));
        }

        try {
            $soap->cancelInvoice($store, $storeId, $order);

            Mage::log(sprintf("Crossroads_Collector: Canceled invoice with collector for order %s in store %s.", $order->getIncrementId(), $storeId));

        }
        catch(Exception $e) {
            Mage::logException($e);

            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage()." (#".$e->getCode().")", $e->getCode(), $e);
        }

        $payment->setIsTransactionClosed(1);

        return $this;
    }

    public function acceptPayment(Mage_Payment_Model_Info $payment) {
        $auth = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);

        if( ! $auth || ! $auth->getTxnId()) {
            Mage::throwException("Crossroads_Collector: Failed to obtain authorization id for order '{$payment->getOrder()->getIncrementId()}' when accepting payment.");
        }

        $payment->setIsTransactionApproved(true);
        $payment->setIsTransactionClosed(true);

        return true;
    }
}
