<?php

class Crossroads_CollectorCheckout_WebhookController extends Mage_Core_Controller_Front_Action
{
    const FIELD_CUSTOMER_CHECKSUM = 'collector_customer_checksum';

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout
     */
    protected function checkoutHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout');
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Config
     */
    protected function configHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/config');
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Data
     */
    protected function helper()
    {
        return Mage::helper('Crossroads_CollectorCheckout');
    }

    /**
     * @param int $code
     * @param $body
     * @return Zend_Controller_Response_Abstract
     * @throws Zend_Controller_Response_Exception
     */
    protected function jsonResponse($code, $body)
    {
        return $this->getResponse()
            ->setHttpResponseCode($code)
            ->setHeader('Content-Type', 'application/json', true)
            ->setBody(json_encode($body));
    }

    /**
     * /Crossroads_CollectorCheckout/webhook/validation
     *
     * Called by collector right before validating the purchase.
     * Check stock here to make sure order can be blacked.
     *
     * @throws Exception
     */
    public function validationAction()
    {
        if ('GET' !== $this->getRequest()->getMethod()) {
            return $this->jsonResponse(405, [
                'title'   => $this->__('Could not place order'),
                'message' => 'Method not allowed',
            ]);
        }

        $privateId = $this->getRequest()->getParam('cid');

        if (!$privateId) {
            return $this->jsonResponse(400, [
                'title'   => $this->__('Could not place order'),
                'message' => 'Missing private checkout id',
            ]);
        }

        try {
            $result = $this->loadCheckoutSessionInformation($privateId);
            $quote = $this->loadAndUpdateQuoteInfo($result);

            $this->updateOrderAddresses($result, $quote);

            // This will call Crossroads_CollectorCheckout_Model_Method_Checkout::authorize
            // Authorize will validate that that there are no changes to the cart that has not been
            // sent to collector
            /** @var Mage_Sales_Model_Order $order */
            $order = Mage::helper('Crossroads_CollectorCheckout/convertQuote')->convert($quote);

            $order->save();

        } catch (Crossroads_API_ResponseException | MageQL_Sales_SubmitOrderException | Mage_Core_Exception $e) {
            Mage::logException($e);

            return $this->jsonResponse(400, [
                // Display the exception message to the customer
                'title' => $this->__('Could not place order'),
                'message' => $e->getMessage(),
            ]);
        } catch (Exception $e) {
            Mage::logException($e);

            return $this->jsonResponse(500, [
                // Exception message could contain sensitive data
                // Don't display to the customer
                'title' => $this->configHelper()->isTest()
                    ? get_class($e)
                    : $this->__('Could not place order'),
                'message' => $this->configHelper()->isTest()
                    ? $e->getMessage()
                    : $this->__('Something went wrong when placing the order'),
            ]);
        }

        if (!$order->getIncrementId()) {
            return $this->jsonResponse(500, [
                'title' => $this->__('Could not place order'),
                'message' => $this->__('Order not generated correctly'),
            ]);
        }

        return $this->jsonResponse(200, [
            'orderReference' => $order->getIncrementId(),
        ]);
    }

    /**
     * Called when something has changed
     *
     * @return Zend_Controller_Response_Abstract
     * @throws Exception
     */
    public function notificationAction()
    {
        if ('GET' !== $this->getRequest()->getMethod()) {
            return $this->jsonResponse(405, ['message' => 'Method not allowed']);
        }

        $privateId = $this->getRequest()->getParam('cid');

        if (!$privateId) {
            return $this->jsonResponse(400, ['message' => 'Missing private checkout id']);
        }

        try {
            $result = $this->loadCheckoutSessionInformation($privateId);

            // This will save the purchaseIdentifier/InvoiceId
            $this->loadAndUpdateQuoteInfo($result);
            $order = $this->loadAndUpdateOrderInfo($result);

            if ($result && $result->data->status === "PurchaseCompleted" && $order->isPaymentReview()) {
                $payment = $order->getPayment();

                if( ! $payment) {
                    throw new Exception(sprintf(
                        "%s: Missing payment information for quote %d",
                        __METHOD__,
                        $order->getId()
                    ));
                }

                $invoiceId = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);

                if( ! $invoiceId) {
                    throw new Exception(sprintf(
                        "%s: Missing invoice id for order %s (%d)",
                        __METHOD__,
                        $order->getIncrementId(),
                        $order->getId()
                    ));
                }

                $this->updateOrderAddresses($result, $order);

                // Approve payment
                $payment->setIsTransactionPending(false);
                $payment->setIsTransactionApproved(true);
                $payment->setTransactionId($invoiceId);
                $payment->registerPaymentReviewAction(Mage_Sales_Model_Order_Payment::REVIEW_ACTION_UPDATE, false);
                $order->setCanSendNewEmailFlag(true);

                $payment->save();
                $order->save();

                try {
                    $order->queueNewOrderEmail();
                }
                catch(Exception $e) {
                    Mage::logException($e);
                }

                if($order->getIsVirtual()) {
                    $payment->capture(null);

                    $payment->save();
                    $order->save();
                }
                else {
                    // Send any virtual products since we have an authorization approved
                    Mage::dispatchEvent('crossroads_order_payment_complete', [ "order" => $order ]);
                }
            }
        } catch (Exception $exception) {
            Mage::logException($exception);

            return $this->jsonResponse(500, ['message' => $exception->getMessage()]);
        }

        return $this->jsonResponse(200, new stdClass());
    }

    /**
     * @param $privateId
     * @return object|null
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function loadCheckoutSessionInformation($privateId)
    {
        $storeId = Mage::app()->getStore()->getStoreId();

        $result = $this->checkoutHelper()->acquireInformationAboutACheckoutSession($privateId, $storeId);

        if (empty($result->data)) {
            throw new Mage_Core_Exception('Unable to get data from collector');
        }

        if (empty($result->data->reference)) {
            throw new Mage_Core_Exception('Missing reference on collector session');
        }

        return $result;
    }

    /**
     * @param $checkoutInformation
     * @return Mage_Sales_Model_Quote
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function loadAndUpdateQuoteInfo($checkoutInformation)
    {
        $quote = $this->helper()->loadQuoteWithReservedId($checkoutInformation->data->reference);

        if (empty($quote)) {
            throw new Mage_Core_Exception('Quote does not exists');
        }

        $this->checkoutHelper()->saveInfoOnPayment($quote->getPayment(), $checkoutInformation->data);

        return $quote;
    }

    /**
     * @param $checkoutInformation
     * @return Mage_Sales_Model_Order
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function loadAndUpdateOrderInfo($checkoutInformation)
    {
        $order = $this->helper()->loadOrderByIncrementId($checkoutInformation->data->reference);

        if (empty($order)) {
            throw new Mage_Core_Exception('Order does not exists');
        }

        $this->checkoutHelper()->saveInfoOnPayment($order->getPayment(), $checkoutInformation->data);

        return $order;
    }

    /**
     * @param object $checkoutInformation
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $order
     */
    private function updateOrderAddresses(object $checkoutInformation, Mage_Core_Model_Abstract $order): void
    {
        $oldBillingAddress = $order->getBillingAddress();
        $oldShippingAddress = $order->getShippingAddress();
        $payment = $order->getPayment();
        $store = Mage::app()->getStore($order->getStoreId());
        $defaultCountryId = $this->getDefaultCountryId($store);

        $billingAddress = array(
            'firstname' => $checkoutInformation->data->customer->billingAddress->firstName,
            'lastname' => $checkoutInformation->data->customer->billingAddress->lastName,
            'city' => $checkoutInformation->data->customer->billingAddress->city,
            'street' => implode("\n", array_filter(array_map("trim", [
                $checkoutInformation->data->customer->billingAddress->address,
                $checkoutInformation->data->customer->billingAddress->address2 ?? "",
            ]))),
            'country_id' => $defaultCountryId,
            'postcode' => $checkoutInformation->data->customer->billingAddress->postalCode,
            'telephone' => $checkoutInformation->data->customer->mobilePhoneNumber,
        );

        $shippingAddress = array(
            'firstname' => $checkoutInformation->data->customer->deliveryAddress->firstName,
            'lastname' => $checkoutInformation->data->customer->deliveryAddress->lastName,
            'city' => $checkoutInformation->data->customer->deliveryAddress->city,
            'street' => implode("\n", array_filter(array_map("trim", [
                $checkoutInformation->data->customer->deliveryAddress->address,
                $checkoutInformation->data->customer->deliveryAddress->address2 ?? "",
            ]))),
            'country_id' => $defaultCountryId,
            'postcode' => $checkoutInformation->data->customer->deliveryAddress->postalCode,
            'telephone' => $checkoutInformation->data->customer->mobilePhoneNumber,
        );

        $oldBillingAddress->addData($billingAddress);

        if($oldShippingAddress) {
            $oldShippingAddress->addData($shippingAddress);
        }

        $order->setCustomerEmail($checkoutInformation->data->customer->email);

        // Update customerinfo on payment to not throw error in 'salesQuoteCollectTotalsAfter'
        $customerBody = [
            'email' => $checkoutInformation->data->customer->email,
            'mobilePhoneNumber' => $checkoutInformation->data->customer->mobilePhoneNumber,
        ];

        $payment->setAdditionalInformation(
            self::FIELD_CUSTOMER_CHECKSUM,
            md5(json_encode($customerBody))
        );

        // Save changes
        $payment->save();
        $oldBillingAddress->save();

        if($oldShippingAddress) {
            $oldShippingAddress->save();
        }

        $order->save();
    }

    private function getDefaultCountryId(Mage_Core_Model_Store $store): string {
        $countryId = $store->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

        if( ! $countryId) {
            throw new Exception(sprintf(
                "%s: The default country is not set for store '%s'.",
                __METHOD__,
                $store->getCode()
            ));
        }

        return $countryId;
    }
}
