<?php

class Crossroads_CollectorCheckout_Model_Method_Checkout extends Mage_Payment_Model_Method_Abstract
{
    protected $_code = 'Crossroads_CollectorCheckout';

    protected $_isGateway = true;

    protected $_isOnline = true;

    protected $_canUseForMultishipping = false;

    protected $_canUseInternal = true;

    protected $_canAuthorize = true;

    protected $_canRefund = false;

    protected $_canCapture = true;

    protected $_canVoid = true;

    protected $_canCancel = true;

    protected $_canFetchTransactionInfo = true;

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout
     */
    protected function checkoutHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout');
    }

    /**
     * Check whether payment method can be used
     *
     * @param Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        $storeId = $quote ? $quote->getStoreId() : null;

        return parent::isAvailable($quote)
            && null !== $quote
            && $this->config()->merchantStoreId($storeId)
            && $this->config()->userName($storeId)
            && $this->config()->sharedAccessKey($storeId)
            && $quote->getBaseGrandTotal() >= $this->config()->minOrderTotal($storeId);
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param Mage_Payment_Model_Info|Varien_Object $payment
     * @param float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function authorize(Varien_Object $payment, $amount)
    {
        // Should only be authorized by a Collector webhook callback.
        // Not by customer order submit
        $this->validateIsCollectorCallback($payment);

        // Make sure that there are no modifications to the cart or fees that has not been
        // Sent to collector.
        $this->validateNoModifications($payment);

        // Leave the transaction opened so it can be captured later
        $payment->setIsTransactionClosed(false);
        // It has to be pending to put it in pending payment
        $payment->setIsTransactionPending(true);

        // We have just made a shallow authorization, so the customer might not
        // approve the final confirmation. Do not send the order confirmation
        // email yet.
        $payment->getOrder()->setCanSendNewEmailFlag(false);

        $payment->save();

        return $this;
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @throw Crossroads_API_ResponseException
     */
    protected function validateIsCollectorCallback(Mage_Payment_Model_Info $payment)
    {
        $isCallback = Mage::app()->getRequest()->getParam('cid')
            === $this->checkoutHelper()->getPrivateCheckoutId($payment);

        if (!$isCallback) {
            throw Crossroads_API_ResponseException::create(400, 'Not allowed to perform manual checkout with Collector Checkout');
        }
    }

    public function canCapture()
    {
        $payment = $this->getInfoInstance();

        if (!$payment) {
            return false;
        }

        $invoiceStatus = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS);

        return Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY === $invoiceStatus;
    }

    /**
     * Activate the invoice.
     *
     * @param Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @param float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function capture(Varien_Object $payment, $amount)
    {
        /** @var Crossroads_Collector_Helper_Soap $soap */
        $soap = Mage::helper('Crossroads_Collector/soap');
        $order = $payment->getOrder();
        $invoiceId = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);

        if (!$invoiceId) {
            throw new Mage_Core_Exception('Unable to capture. Missing purchaseIdentifier on payment.');
        }

        $merchantStoreId = $this->config()->merchantStoreId();

        try {
            $result = $soap->captureInvoice($order->getStore(), $merchantStoreId, $order);

            Mage::log(sprintf('Crossroads_Collector: Captured invoice with collector for order %s in store %s, got payment reference %s',
                $order->getIncrementId(), $merchantStoreId, $result->PaymentReference));

            // If we have a response, it is successful and the invoice has been activated
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS,
                Crossroads_Collector_Helper_Data::INVOICE_STATUS_ACTIVATED
            );
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_PAYMENT_REFERENCE,
                $result->PaymentReference
            );
        } catch (Exception $e) {
            Mage::logException($e);

            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage().' (#'.$e->getCode().')', $e->getCode(), $e);
        }

        $order->setCanSendNewEmailFlag(true);
        $payment->setParentTransactionId($invoiceId);
        $payment->setTransactionId($invoiceId);
        $payment->setIsTransactionPending(0);
        $payment->setIsTransactionClosed(1);
        $payment->save();

        // If payment is captured, issue serialcodes and retain24-vouchers
        Mage::dispatchEvent('crossroads_order_payment_complete', [ "order" => $order ]);

        return $this;
    }

    /**
     * Cancels a reservation with Collector.
     *
     * @param Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function cancel(Varien_Object $payment)
    {
        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        return $this;
    }

    /**
     * @param Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function void(Varien_Object $payment)
    {
        return $this->cancel($payment);
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_CollectorCheckout/config');
    }

    /**
     * @param Mage_Payment_Model_Info|Varien_Object $payment
     * @throws Exception
     */
    private function validateNoModifications(Varien_Object $payment)
    {
        if ($payment instanceof Mage_Sales_Model_Quote_Payment) {
            $quote = $payment->getQuote();
        } else {
            $quoteId = $payment->getOrder()->getQuoteId();

            $quote = Mage::getModel('sales/quote')->load($quoteId);
        }

        $privateId = $this->checkoutHelper()->getPrivateCheckoutId($payment);

        if (!$privateId) {
            throw Crossroads_API_ResponseException::create(500, 'Unable to get private checkout id');
        }

        if ($this->checkoutHelper()->hasCartChanges($quote)) {
            throw Crossroads_API_ResponseException::create(500, 'Collector Bank has not been updated with the latest changes to the cart');
        }

        if ($this->checkoutHelper()->hasFeesChanges($quote)) {
            throw Crossroads_API_ResponseException::create(500, 'Collector Bank has not been updated with the latest changes to fees');
        }
    }
}
