<?php

class Crossroads_CollectorCheckout_Helper_Checkout_RequestBuilder extends Mage_Core_Helper_Abstract
{
    /**
     * Event triggered in cartBody after all items have been added from quote
     *
     * Params:
     * * quote: Mage_Sales_Model_Quote
     * * container: Varien_Object
     */
    const EVENT_CART_BODY_POST_ADD_ITEMS = "crossroads_collector_checkout_cart_body_post_add_items";

    const PROFILE_DIGITAL = "Digital";

    /**
     * @return Crossroads_CollectorCheckout_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_CollectorCheckout/config');
    }

    /**
     * @see https://checkout-documentation.collector.se/#1-initialize-a-checkout-session
     * @param Mage_Sales_Model_Quote $quote
     * @param $storeId
     * @param $countryCode
     * @param $callbackUrl
     * @return array
     */
    public function checkoutInitializationBody(Mage_Sales_Model_Quote $quote)
    {
        $config = $this->config();

        $store = $quote->getStore();

        $body = [
            // Received from Collector Merchant Services.
            'storeId'     => $config->merchantStoreId($quote->getStoreId()),

            // The country code to use. SE, NO, FI, DK and DE is supported.
            'countryCode' => $config->countryCode($quote->getStoreId()),

            // A reference to the order, i.e. order ID or similar. Max 50 chars.
            'reference'   => $quote->getReservedOrderId(),

            'merchantTermsUri' => $config->termsUri($quote->getStoreId()),
            'notificationUri'  => $config->getCallbackUrl($store, 'Crossroads_CollectorCheckout/webhook/notification').'?cid={checkout.id}',
            'validationUri'    => $config->getCallbackUrl($store, 'Crossroads_CollectorCheckout/webhook/validation').'?cid={checkout.id}',

            'cart'     => $this->cartBody($quote),
            'fees'     => $this->feesBody($quote),
        ];

        $customerBody = $this->customerBody($quote);

        if (!empty($customerBody["email"]) && !empty($customerBody["mobilePhoneNumber"])) {
            $body['customer'] = $customerBody;
        }

        if ($profileName = $this->getProfileName($quote)) {
            $body['profileName'] = $profileName;
        }

        return $body;
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return string|null
     */
    public function getProfileName(Mage_Sales_Model_Quote $quote) {
        if ($this->config()->alwaysUseDigital($quote->getStoreId())) {
            return static::PROFILE_DIGITAL;
        }

        if ($quote->isVirtual() && $this->customerBody($quote)) {
            return static::PROFILE_DIGITAL;
        }

        return null;
    }

    /**
     * Discounts should have its own order row with a negative amount.
     *
     * @param Mage_Sales_Model_Quote $quote
     * @return array
     */
    public function cartBody(Mage_Sales_Model_Quote $quote)
    {
        $items = [];

        $total = 0;

        /** @var Mage_Sales_Model_Quote_Item $item */
        foreach ($quote->getAllVisibleItems() as $item) {
            // Collect for bundle products
            $childDiscount = 0.0;

            /** @var Mage_Sales_Model_Quote_Item $child */
            foreach ($item->getChildren() as $child) {
                $childDiscount += (float) $child->getBaseDiscountAmount();
            }

            $items[] = [
                // Required.
                // The article id or equivalent.
                // Max 50 characters.
                // The combination of id and description (property below) must be unique
                // within the checkout session including the fees in the cart.
                // Shown on the invoice or receipt.
                'id'                   => (string) $item->getProduct()->getId(),

                // Required.
                // Descriptions longer than 50 characters will be truncated.
                // Shown on the invoice or receipt.
                'description'          => $item->getName(),

                // Required.
                // The unit price of the article including VAT.
                // Positive and negative values allowed. Max 2 decimals, i.e. 100.00
                'unitPrice'            => (float) $item->getPriceInclTax(), // Per item

                // Required.
                // Quantity of the article. Allowed values are 1 to 99999999
                'quantity'             => $item->getQty(),

                // Required.
                // The VAT of the article in percent. Allowed values are 0 to 100. Max 2 decimals, i.e. 25.00
                'vat'                  => round($item->getTaxPercent(), 2),

                // Optional.
                // When set to true it indicates that a product needs strong identification
                // and the customer will need to strongly identify themselves
                // at the point of purchase using Mobilt BankID.
                'requiresElectronicId' => $this->requiresElectronicId($item),

                // Optional.
                // Maximum allowed characters are 1024.
                'sku'                  => $item->getSku(),
            ];

            if ((float) $item->getBaseDiscountAmount() || $childDiscount) {
                $items[] = [
                    'id'          => $item->getId().'-discount',
                    'description' => Mage::helper('core')->__('Discount').': '.$item->getName(),
                    // total discount for row
                    'unitPrice'   => -((float) $item->getBaseDiscountAmount() + $childDiscount),
                    'quantity'    => 1,
                    'vat'         => round($item->getTaxPercent(), 2),
                ];
            }

            $total += $item->getPriceInclTax() * $item->getQty() - $item->getBaseDiscountAmount() - $childDiscount;
        }

        $container = new Varien_Object();
        $container->setItems($items);

        Mage::dispatchEvent(self::EVENT_CART_BODY_POST_ADD_ITEMS, [
            "quote"  => $quote,
            "container" => $container
        ]);

        return [
            'items' => $container->getItems(),
        ];
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return object
     */
    public function feesBody(Mage_Sales_Model_Quote $quote)
    {
        $fees = []; // Fees must be assoc

        /** @var Mage_Sales_Model_Quote_Address_Total[] $totals */
        $totals = $quote->getTotals();

        // Payment fee
        if (isset($totals['payment_fee']) && ($price = (float) $totals['payment_fee']->getValueInclTax())) {
            $exlTax = (float) $totals['payment_fee']->getValueExclTax();

            $tax = $price - $exlTax;
            $taxPercent = $tax / ($price - $tax);

            $fees['payment_fee'] = [
                'id'          => $totals['payment_fee']->getCode(),
                'description' => $totals['payment_fee']->getTitle(),
                'unitPrice'   => $price,
                'quantity'    => 1,
                'vat'         => round($taxPercent * 100, 2),
            ];
        }

        $address = $quote->getShippingAddress();

        if ($address->getBaseShippingInclTax() > 0) {
            // Note: discount can't be separate row because fees cant be negative
            $incTax = (float) $address->getBaseShippingInclTax() - (float) $address->getBaseShippingDiscountAmount();
            $exlTax = $incTax - $address->getShippingTaxAmount();

            $fees['shipping'] = [
                'id'          => $address->getShippingMethod(),
                'description' => $address->getShippingDescription(),
                'unitPrice'   => $incTax,
                'vat'         => round(($address->getShippingTaxAmount() / $exlTax) * 100, 2),
            ];
        }

        // Todo: add integration with Crossroads_Fees

        return (object) $fees; // Has to be converted to object in case it is empty
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return array|null
     */
    public function customerBody(Mage_Sales_Model_Quote $quote)
    {
        if (!$quote->getCustomerEmail() || !$quote->getBillingAddress()->getTelephone()) {
            return null;
        }

        return [
            'email'             => $quote->getCustomerEmail(),
            'mobilePhoneNumber' => $quote->getBillingAddress()->getTelephone(),
        ];
    }

    /**
     * Should always be true for virtual products.
     * Note that configurable and bundled products can contain virtual products.
     *
     * @param Mage_Sales_Model_Quote_Item $item
     * @return bool
     */
    private function requiresElectronicId(Mage_Sales_Model_Quote_Item $item)
    {
        if ($item->getIsVirtual()) {
            return true;
        }

        $product = Mage::getModel('catalog/product')->load($item->getProduct()->getId());

        if ('bundle' === $item->getProduct()->getTypeId()) {
            $collection = $product->getTypeInstance(true)
                ->getSelectionsCollection($product->getTypeInstance(true)->getOptionsIds($product), $product);
            foreach ($collection as $child) {
                if ($child->getIsVirtual()) {
                    return true;
                }
            }
        }

        return false;
    }
}
