<?php

class Crossroads_CollectorCheckout_Controller_Payment extends Crossroads_API_Controller_Super
{
    /**
     * @return Mage_Sales_Model_Quote
     */
    public function getQuote()
    {
        return Mage::getSingleton('checkout/session')->getQuote();
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout
     */
    protected function checkoutHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout');
    }

    /**
     * Get the cached collector checkout info on the order.
     * Can be used to detect if info has been changed by a web hook.
     * if response.status = 'PurchaseCompleted' then the cart is locked for modification and
     * the checkout should be finished asap.
     *
     * /api/payment/Crossroads_CollectorCheckout/poll
     *
     * @throws Exception
     */
    public function pollAction()
    {
        $quote = $this->getQuote();

        if (!$quote) {
            throw Crossroads_API_ResponseException::create(400, 'No quote');
        }

        $info = $this->checkoutHelper()->getInfo($quote->getPayment());

        if (!$info) {
            return $this->sendData([200, ['status' => 'NotInitialized']]);
        }

        return $this->sendData([200, $info]);
    }

    /**
     * /api/payment/Crossroads_CollectorCheckout/cancel
     *
     * @throws Exception
     */
    public function cancelAction()
    {
        $quote = $this->getQuote();

        if (!$quote) {
            throw Crossroads_API_ResponseException::create(400, 'No quote');
        }

        $this->checkoutHelper()->cancel($quote->getPayment());

        return $this->sendData([200, ['message' => 'Reset']]);
    }

    /**
     * Fetch and refresh the collector checkout status of a quote. Useful for debugging.
     *
     * /api/payment/Crossroads_CollectorCheckout/info
     *
     * @throws Exception
     */
    public function infoAction()
    {
        $quote = $this->getQuote();

        if (!$quote) {
            throw Crossroads_API_ResponseException::create(400, 'No quote');
        }

        $privateId = $this->checkoutHelper()->getPrivateCheckoutId($quote->getPayment());

        if (!$privateId) {
            throw Crossroads_API_ResponseException::create(400, 'Collector not initialized');
        }

        try {
            $storeId = Mage::app()->getStore()->getStoreId();

            $result = $this->checkoutHelper()->acquireInformationAboutACheckoutSession($privateId, $storeId);
        } catch (Exception $exception) {
            return Crossroads_API_ResponseException::create(500, $exception->getMessage());
        }

        return $this->sendData([200, $result->data]);
    }
}
