<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;


class Crossroads_CollectorCheckout_Model_Graphql_Payment extends MageQL_Sales_Model_Payment_Abstract {
    const INITIALIZED = "Initialized";
    const CUSTOMER_IDENTIFIED = "CustomerIdentified";
    const COMMITED_TO_PURCHASE = "CommittedToPurchase";
    const PURCHASE_COMPLETED = "PurchaseCompleted";
    const NOT_INITIALIZED = "NotInitialized";

    public function getCode(): string {
        return "Crossroads_CollectorCheckout";
    }

    public function getMethodTypeName(): string {
        return "CollectorCheckout";
    }

    /**
     * @return Array<FieldBuilder>
     */
    public function getPaymentMethodFields(Registry $registry): array {
        return [];
    }

    public function getQuotePaymentFields(Registry $registry): array {
        return [
            "publicToken" => $this->field("ID!", "The public token for Collector Checkout")
                ->setResolver([$this, "resolvePublicToken"]),
            "status" => $this->field("CollectorCheckoutPaymentStatus!", "The status of the Collector Checkout payment")
                ->setResolver([$this, "resolvePaymentStatus"]),
        ];
    }

    public function getMutateSetQuotePaymentFields(Registry $registry): array {
        return [
            "result" => $this->field(
                $this->getMutationResultType()."!",
                "Result of attempting to set the payment method to Collector Checkout"
            )
                // Simple result, we either succeed or fail
                ->setResolver("MageQL\\forwardResolver"),
            "publicToken" => $this->field("ID!", "The public token for Collector Checkout")
                ->setResolver([$this, "resolvePublicToken"]),
        ];
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
            case "CollectorCheckoutPaymentStatus":
                return $this->enum("Type of status of the Collector Checkout payment", [
                    self::INITIALIZED => [
                        "description" => "Checkout is started but no information has been supplied by the customer",
                    ],
                    self::CUSTOMER_IDENTIFIED => [
                        "description" => "Customer has been successfully identified at Collector",
                    ],
                    self::COMMITED_TO_PURCHASE => [
                        "description" => "The customer has pressed the purchase button and the system is waiting for a successful/unsuccessful outcome",
                    ],
                    self::NOT_INITIALIZED => [
                        "description" => "Payment have not yet been initialized",
                    ],
                    self::PURCHASE_COMPLETED => [
                        "description" => "The purchase has been completed",
                    ],
                ]);


            default:
                return parent::getTypeBuilder($typeName, $registry);
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        return parent::getTypeFields($typeName, $registry);
    }


    public function resolvePaymentStatus(Mage_Sales_Model_Quote_Payment $payment, array $args, MageQL_Core_Model_Context $ctx) {
        $helper = Mage::helper("Crossroads_CollectorCheckout/checkout");

        $info = $helper->getInfo($payment);

        if (!$info) {
            return self::NOT_INITIALIZED;
        }

        return $info["status"];
    }

    public function resolvePublicToken($src, array $args, MageQL_Core_Model_Context $ctx) {
        $helper = Mage::helper("Crossroads_CollectorCheckout/checkout");

        return $helper->getOrCreatePublicToken();
    }

    public function mutateSetPaymentMethod(Mage_Sales_Model_Quote $quote, array $args, $ctx, ResolveInfo $info) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $session = Mage::getSingleton("checkout/session");

        try {
            $model->importPaymentData($quote, "Crossroads_CollectorCheckout", []);
            $model->saveSessionQuote();

            // Reset old session data in case we had a previous order
            $session->setLastOrderId(null);
            $session->setLastQuoteId(null);
            $session->setLastSuccessQuoteId(null);
            $session->setLastOrderId(null);
            $session->setLastRealOrderId(null);

            return self::SUCCESS;
        }
        catch(Mage_Core_Exception $e) {
            return $this->translatePaymentError($e);
        }
    }
}