<?php

class Crossroads_CollectorCheckout_WebhookController extends Mage_Core_Controller_Front_Action
{
    const FIELD_CUSTOMER_CHECKSUM   = 'collector_customer_checksum';

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout
     */
    protected function checkoutHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout');
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Data
     */
    protected function helper()
    {
        return Mage::helper('Crossroads_CollectorCheckout');
    }

    /**
     * @param int $code
     * @param $body
     * @return Zend_Controller_Response_Abstract
     * @throws Zend_Controller_Response_Exception
     */
    protected function jsonResponse($code, $body)
    {
        return $this->getResponse()
            ->setHttpResponseCode($code)
            ->setHeader('Content-Type', 'application/json', true)
            ->setBody(json_encode($body));
    }

    /**
     * /Crossroads_CollectorCheckout/webhook/validation
     *
     * Called by collector right before validating the purchase.
     * Check stock here to make sure order can be blacked.
     *
     * @throws Exception
     */
    public function validationAction()
    {
        if ('GET' !== $this->getRequest()->getMethod()) {
            return $this->jsonResponse(405, ['message' => 'Method not allowed']);
        }

        $privateId = $this->getRequest()->getParam('cid');

        if (!$privateId) {
            return $this->jsonResponse(400, ['message' => 'Missing private checkout id']);
        }

        try {
            $quote = $this->loadQuoteUsingPrivateIdAndRefreshInfo($privateId);
        } catch (Exception $exception) {
            return $this->jsonResponse(500, ['message' => $exception->getMessage()]);
        }

        /** @var Mage_Catalog_Model_Product[] $cartItems */
        $cartItems = $quote->getAllItems();

        if (0 === count($cartItems)) {
            return $this->jsonResponse(500, ['message' => 'Cart has been emptied']);
        }

        foreach ($cartItems as $item) {
            $product = Mage::getModel('catalog/product')->load($item->getProductId());

            if (!$product->isInStock()) {
                return $this->jsonResponse(500, ['message' => 'One or more items in cart is no longer available']);
            }
        }

        return $this->jsonResponse(200, [
            'orderReference' => $quote->getReservedOrderId(),
        ]);
    }

    /**
     * Called when something has changed
     *
     * @return Zend_Controller_Response_Abstract
     * @throws Exception
     */
    public function notificationAction()
    {
        if ('GET' !== $this->getRequest()->getMethod()) {
            return $this->jsonResponse(405, ['message' => 'Method not allowed']);
        }

        $privateId = $this->getRequest()->getParam('cid');

        if (!$privateId) {
            return $this->jsonResponse(400, ['message' => 'Missing private checkout id']);
        }

        try {
            $result = $this->loadCheckoutSessionInformation($privateId);
            $quote = $this->loadAndUpdateQuoteInfo($result);
        } catch (Exception $exception) {
            return $this->jsonResponse(500, ['message' => $exception->getMessage()]);
        }

        if ($result->data->status == "PurchaseCompleted") {
            $this->updateQuoteAddresses($result, $quote);
        }

        return $this->jsonResponse(200, new stdClass());
    }

    /**
     * @param $privateId
     * @return Mage_Sales_Model_Quote
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function loadQuoteUsingPrivateIdAndRefreshInfo($privateId)
    {
        $result = $this->loadCheckoutSessionInformation($privateId);

        return $this->loadAndUpdateQuoteInfo($result);
    }

    /**
     * @param $privateId
     * @return object|null
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function loadCheckoutSessionInformation($privateId)
    {
        $storeId = Mage::app()->getStore()->getStoreId();

        $result = $this->checkoutHelper()->acquireInformationAboutACheckoutSession($privateId, $storeId);

        if (empty($result->data)) {
            throw new Mage_Core_Exception('Unable to get data from collector');
        }

        if (empty($result->data->reference)) {
            throw new Mage_Core_Exception('Missing reference on collector session');
        }

        return $result;
    }


    /**
     * @param $checkoutInformation
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function loadAndUpdateQuoteInfo($checkoutInformation)
    {
        $quote = $this->helper()->loadQuoteWithReservedId($checkoutInformation->data->reference);

        if (empty($quote)) {
            throw new Mage_Core_Exception('Quote does not exists');
        }

        $this->checkoutHelper()->saveInfoOnPayment($quote->getPayment(), $checkoutInformation->data);

        return $quote;
    }

    /**
     * @param $checkoutInformation
     */
    private function updateQuoteAddresses($checkoutInformation, $quote)
    {
        $oldBillingAddress = $quote->getBillingAddress();
        $oldShippingAddress = $quote->getShippingAddress();
        $payment = $quote->getPayment();

        $billingAddress = array(
            'firstname' => $checkoutInformation->data->customer->billingAddress->firstName,
            'lastname' => $checkoutInformation->data->customer->billingAddress->lastName,
            'city' => $checkoutInformation->data->customer->billingAddress->city,
            'street' => implode("\n", array_filter(array_map("trim", [
                $checkoutInformation->data->customer->billingAddress->address,
                $checkoutInformation->data->customer->billingAddress->address2 ?? "",
            ]))),
            'country_id' => "SE",
            'postcode' => $checkoutInformation->data->customer->billingAddress->postalCode,
            'telephone' => $checkoutInformation->data->customer->mobilePhoneNumber,
        );

        $shippingAddress = array(
            'firstname' => $checkoutInformation->data->customer->deliveryAddress->firstName,
            'lastname' => $checkoutInformation->data->customer->deliveryAddress->lastName,
            'city' => $checkoutInformation->data->customer->deliveryAddress->city,
            'street' => implode("\n", array_filter(array_map("trim", [
                $checkoutInformation->data->customer->deliveryAddress->address,
                $checkoutInformation->data->customer->deliveryAddress->address2 ?? "",
            ]))),
            'country_id' => "SE",
            'postcode' => $checkoutInformation->data->customer->deliveryAddress->postalCode,
            'telephone' => $checkoutInformation->data->customer->mobilePhoneNumber,
        );

        $oldBillingAddress->addData($billingAddress);
        $oldShippingAddress->addData($shippingAddress);
        $quote->setCustomerEmail($checkoutInformation->data->customer->email);

        // Update customerinfo on payment to not throw error in 'salesQuoteCollectTotalsAfter'
        $customerBody = [
            'email'             => $checkoutInformation->data->customer->email,
            'mobilePhoneNumber' => $checkoutInformation->data->customer->mobilePhoneNumber,
        ];

        $payment->setAdditionalInformation(
            self::FIELD_CUSTOMER_CHECKSUM,
            md5(json_encode($customerBody))
        );

        // Save changes
        $payment->save();
        $oldBillingAddress->save();
        $oldShippingAddress->save();
        $quote->save();
    }
}
