<?php

class Crossroads_CollectorCheckout_Model_Method_Checkout extends Mage_Payment_Model_Method_Abstract
{
    protected $_code = 'Crossroads_CollectorCheckout';

    protected $_isGateway = true;

    protected $_isOnline = true;

    protected $_canUseForMultishipping = false;

    protected $_canUseInternal = true;

    protected $_canAuthorize = true;

    protected $_canRefund = false;

    protected $_canCapture = true;

    protected $_canVoid = true;

    protected $_canCancel = true;

    protected $_canFetchTransactionInfo = true;

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout
     */
    protected function checkoutHelper()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout');
    }

    /**
     * Check whether payment method can be used
     *
     * @param Mage_Sales_Model_Quote|null $quote
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        $storeId = $quote ? $quote->getStoreId() : null;

        return parent::isAvailable($quote)
            && null !== $quote
            && $this->config()->merchantStoreId($storeId)
            && $this->config()->userName($storeId)
            && $this->config()->sharedAccessKey($storeId)
            && $quote->getBaseGrandTotal() >= $this->config()->minOrderTotal($storeId);
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment|Varien_Object $payment
     * @param float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function authorize(Varien_Object $payment, $amount)
    {
        $invoiceId = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);

        $this->validatePassedCreditCheck($payment, $invoiceId);
        $this->validateNoModifications($payment);

        $payment->setTransactionId($invoiceId);

        // Leave the transaction opened so it can be captured later
        $payment->setIsTransactionClosed(false);
        $payment->setIsTransactionApproved(true);

        $payment->save();

        return $this;
    }

    /**
     * Activate the invoice.
     *
     * @param Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment|Varien_Object $payment
     * @param float $amount
     * @return $this
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    public function capture(Varien_Object $payment, $amount)
    {
        /** @var Crossroads_Collector_Helper_Soap $soap */
        $soap = Mage::helper('Crossroads_Collector/soap');
        $order = $payment->getOrder();
        $invoiceId = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID);

        $merchantStoreId = $this->config()->merchantStoreId();

        $this->validatePassedCreditCheck($payment, $invoiceId);
        $this->validateNoModifications($payment);

        try {
            $result = $soap->captureInvoice($order->getStore(), $merchantStoreId, $order);

            Mage::log(sprintf('Crossroads_Collector: Captured invoice with collector for order %s in store %s, got payment reference %s',
                $order->getIncrementId(), $merchantStoreId, $result->PaymentReference));

            // If we have a response, it is successful and the invoice has been activated
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS,
                Crossroads_Collector_Helper_Data::INVOICE_STATUS_ACTIVATED
            );
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_PAYMENT_REFERENCE,
                $result->PaymentReference
            );
        } catch (Exception $e) {
            Mage::logException($e);

            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage().' (#'.$e->getCode().')', $e->getCode(), $e);
        }

        $payment->setParentTransactionId($invoiceId);
        $payment->setTransactionId($invoiceId);
        $payment->setIsTransactionPending(0);
        $payment->setIsTransactionClosed(1);
        $payment->save();

        // If payment is captured, issue serialcodes and retain24-vouchers
        Mage::dispatchEvent('crossroads_order_payment_complete', [ "order" => $order ]);

        return $this;
    }

    /**
     * Cancels a reservation with Collector.
     *
     * @param Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function cancel(Varien_Object $payment)
    {
        $payment->setIsTransactionDenied(true);
        $payment->setIsTransactionClosed(true);

        return $this;
    }

    /**
     * @param Mage_Sales_Model_Order_Payment|Varien_Object $payment
     * @return $this
     * @throws Mage_Core_Exception
     */
    public function void(Varien_Object $payment)
    {
        return $this->cancel($payment);
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_CollectorCheckout/config');
    }

    /**
     * @param Varien_Object $payment
     * @param mixed $invoiceId
     * @throws Exception
     */
    private function validatePassedCreditCheck(Varien_Object $payment, $invoiceId)
    {
        $invoiceStatus = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS);

        if (Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY !== $invoiceStatus) {
            // We can only activate preliminary invoices
            // (the other statuses are either waiting on Collector Bank, activated, or rejected).
            throw new Exception(sprintf(
                "Crossroads_Collector: Invoice status for order '%s' is not PRELIMINARY (1) instead it is '%s', cannot authorize/capture payment of invoice '%s'.",
                $payment->getOrder()->getIncrementId(), $invoiceStatus, $invoiceId
            ));
        }
    }

    /**
     * @param Mage_Sales_Model_Order_Payment|Mage_Sales_Model_Quote_Payment|Varien_Object $payment
     * @throws Exception
     */
    private function validateNoModifications(Varien_Object $payment)
    {
        if ($payment instanceof Mage_Sales_Model_Quote_Payment) {
            $quote = $payment->getQuote();
        } else {
            $quote = $payment->getOrder();
        }

        $storeId = $quote->getStore()->getId();

        $privateId = $this->checkoutHelper()->getPrivateCheckoutId($payment);

        if (!$privateId) {
            throw new Exception('Unable to get private checkout id');
        }

        $result = $this->checkoutHelper()->acquireInformationAboutACheckoutSession($privateId, $storeId);

        if (!isset($result->data)) {
            throw new Exception('Unable to read the response from collector');
        }

        if (!isset($result->data->purchase->amountToPay)) {
            throw new Exception('Amount To Pay is not set. Has the order passed the credit check?');
        }

        // Validate that the value in Collector and match
        if ($result->data->purchase->amountToPay != $quote->getGrandTotal()) {
            throw new Exception('Order totals has changed without updating collector session');
        }
    }
}
