<?php

class Crossroads_CollectorCheckout_Helper_Checkout extends Mage_Core_Helper_Abstract
{
    const FIELD_PUBLIC_TOKEN        = 'collector_public_token';
    const FIELD_PUBLIC_TOKEN_EXPIRE = 'collector_public_token_expire';
    const FIELD_PRIVATE_ID          = 'collector_private_id';
    const FIELD_INFO                = 'collector_session_info';
    const FIELD_CART_CHECKSUM       = 'collector_cart_checksum';
    const FIELD_FEES_CHECKSUM       = 'collector_fees_checksum';
    const FIELD_CUSTOMER_CHECKSUM   = 'collector_customer_checksum';

    /**
     * @return Mage_Sales_Model_Quote
     */
    public function getQuote()
    {
        return Mage::getSingleton('checkout/session')->getQuote();
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout_ApiClient
     */
    protected function apiClient()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout_apiClient');
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Checkout_RequestBuilder
     */
    protected function requestBuilder()
    {
        return Mage::helper('Crossroads_CollectorCheckout/checkout_requestBuilder');
    }

    /**
     * @return Crossroads_CollectorCheckout_Helper_Config
     */
    protected function config()
    {
        return Mage::helper('Crossroads_CollectorCheckout/config');
    }

    /**
     * @return string|null
     * @throws Mage_Core_Exception
     */
    public function getOrCreatePublicToken()
    {
        return $this->getPublicToken() ?: $this->initializeCheckoutSession();
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @throws Mage_Core_Exception
     */
    public function cancel($payment)
    {
        $payment->setAdditionalInformation(static::FIELD_PRIVATE_ID, null);
        $payment->setAdditionalInformation(static::FIELD_PUBLIC_TOKEN, null);
        $payment->setAdditionalInformation(static::FIELD_PUBLIC_TOKEN_EXPIRE, null);
        $payment->setAdditionalInformation(static::FIELD_CART_CHECKSUM, null);
        $payment->setAdditionalInformation(static::FIELD_FEES_CHECKSUM, null);
        $payment->setAdditionalInformation(static::FIELD_CUSTOMER_CHECKSUM, null);
        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS, null);

        $payment->save();
    }

    /**
     * Returns null if no token or if it has expired.
     *
     * @return string|null
     */
    public function getPublicToken()
    {
        $payment = $this->getQuote()->getPayment();

        $token = $payment->getAdditionalInformation(static::FIELD_PUBLIC_TOKEN);
        $expire = $payment->getAdditionalInformation(static::FIELD_PUBLIC_TOKEN_EXPIRE);

        // Timestamp 7 days after token was created. Example format: 2019-07-02T15:39:12.8929039+00:00
        if ($token && $expire && strtotime($expire) > time()) {
            return $token;
        }

        return null;
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @return string|null
     */
    public function getPrivateCheckoutId($payment)
    {
        return $payment->getAdditionalInformation(static::FIELD_PRIVATE_ID);
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @return mixed
     */
    public function getStoreId($payment)
    {
        return $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_STORE_ID);
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @return mixed
     */
    public function getInfo($payment)
    {
        $data = $payment->getAdditionalInformation(static::FIELD_INFO);

        return json_decode($data, true);
    }

    /**
     * @param string $privateId
     * @param null $storeId
     * @return object|null
     * @throws Exception
     */
    public function acquireInformationAboutACheckoutSession($privateId, $storeId = null)
    {
        $storeId = $this->config()->merchantStoreId($storeId);

        return $this->apiClient()->get("merchants/$storeId/checkouts/$privateId");
    }

    /**
     * @param Mage_Payment_Model_Info $payment
     * @param stdClass $data
     * @throws Mage_Core_Exception
     */
    public function saveInfoOnPayment($payment, stdClass $data)
    {
        if ($data->purchase && $data->purchase->purchaseIdentifier) {
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID,
                $data->purchase->purchaseIdentifier
            );
        }

        if ($data->purchase && 'Preliminary' === $data->purchase->result) {
            $payment->setAdditionalInformation(
                Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS,
                Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY
            );
        }

        /** @var Mage_Sales_Model_Quote|Mage_Sales_Model_Order $quote */
        $quote = $payment->getOrder() ?: $payment->getQuote();

        $merchantStoreId = $this->config()
            ->merchantStoreId($quote->getStoreId());

        $payment->setAdditionalInformation(
            Crossroads_Collector_Helper_Data::FIELD_STORE_ID,
            $merchantStoreId
        );

        // These two are not really used by this payment method at this moment.
        // In collector invoice module it is used to figure out what store id to use, but we need to define
        // The store id when initializing the checkout and can not change it.
        $payment->setAdditionalInformation(
            Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY,
            (bool) $data->businessCustomer // NULL when private customer
        );
        $payment->setAdditionalInformation(
            Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK,
            false
        );

        $payment->setAdditionalInformation(static::FIELD_INFO, json_encode($data));

        $payment->save();
    }

    /**
     * @return string
     * @throws Mage_Core_Exception
     * @throws Exception
     */
    private function initializeCheckoutSession()
    {
        $quote = $this->getQuote();

        if (!$quote->getReservedOrderId()) {
            $quote->reserveOrderId();
            $quote->save();
        }

        $body = $this->requestBuilder()->checkoutInitializationBody($quote);

        $result = $this->apiClient()->post('checkout', $body);

        $payment = $this->getQuote()->getPayment();

        $payment->setAdditionalInformation(static::FIELD_PUBLIC_TOKEN, $result->data->publicToken);
        $payment->setAdditionalInformation(static::FIELD_PUBLIC_TOKEN_EXPIRE, $result->data->expiresAt);
        $payment->setAdditionalInformation(static::FIELD_PRIVATE_ID, $result->data->privateId);

        $payment->setAdditionalInformation(static::FIELD_CART_CHECKSUM, md5(json_encode($body['cart'])));
        $payment->setAdditionalInformation(static::FIELD_FEES_CHECKSUM, md5(json_encode($body['fees'])));

        if (isset($body["customer"])) {
            $payment->setAdditionalInformation(static::FIELD_CUSTOMER_CHECKSUM, md5(json_encode($body['customer'])));
        }
        else {
            $payment->setAdditionalInformation(static::FIELD_CUSTOMER_CHECKSUM, "");
        }

        $payment->setAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS, null);
        $payment->setAdditionalInformation(
            Crossroads_Collector_Helper_Data::FIELD_STORE_ID,
            $this->config()->merchantStoreId()
        );

        $payment->save();

        return $result->data->publicToken;
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return bool
     */
    public function hasCartChanges($quote)
    {
        $body = $this->requestBuilder()->cartBody($quote);
        $checksum = md5(json_encode($body));

        return $checksum !== $quote->getPayment()->getAdditionalInformation(static::FIELD_CART_CHECKSUM);
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return bool
     */
    public function hasFeesChanges($quote)
    {
        $body = $this->requestBuilder()->feesBody($quote);
        $checksum = md5(json_encode($body));

        return $checksum !== $quote->getPayment()->getAdditionalInformation(static::FIELD_FEES_CHECKSUM);
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return bool
     */
    public function hasCustomerChanges($quote)
    {
        $body = $this->requestBuilder()->customerBody($quote);

        if (!empty($body["email"]) && !empty($body["mobilePhoneNumber"])) {
            $checksum = md5(json_encode($body));
        }
        else {
            $checksum = "";
        }

        return $checksum !== $quote->getPayment()->getAdditionalInformation(static::FIELD_CUSTOMER_CHECKSUM);
    }

    /**
     * @see https://checkout-documentation.collector.se/#update-cart
     * @param Mage_Sales_Model_Quote $quote
     * @throws Exception
     */
    public function updateCart($quote)
    {
        $body = $this->requestBuilder()->cartBody($quote);
        $checksum = md5(json_encode($body));

        $privateId = $this->getPrivateCheckoutId($quote->getPayment());
        $storeId = $this->getStoreId($quote->getPayment());

        $quote->getPayment()->setAdditionalInformation(static::FIELD_CART_CHECKSUM, $checksum)
            ->save();

        $this->apiClient()->put("merchants/$storeId/checkouts/$privateId/cart", $body);
    }

    /**
     * @see https://checkout-documentation.collector.se/#update-cart
     * @param Mage_Sales_Model_Quote $quote
     * @throws Exception
     */
    public function updateFees($quote)
    {
        $body = $this->requestBuilder()->feesBody($quote);
        $checksum = md5(json_encode($body));

        $privateId = $this->getPrivateCheckoutId($quote->getPayment());
        $storeId = $this->getStoreId($quote->getPayment());

        $quote->getPayment()->setAdditionalInformation(static::FIELD_FEES_CHECKSUM, $checksum)
            ->save();

        $this->apiClient()->put("merchants/$storeId/checkouts/$privateId/fees", $body);
    }

    /**
     * Not possible. If we detect a change then we reset so that a new public token will be requested.
     *
     * @param Mage_Sales_Model_Quote $quote
     * @throws Mage_Core_Exception
     */
    public function updateCustomer(Mage_Sales_Model_Quote $quote)
    {
        $this->cancel($quote->getPayment());
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return bool
     */
    public function quoteUsesThisPaymentMethod(Mage_Sales_Model_Quote $quote)
    {
        $method = Mage::getModel('Crossroads_CollectorCheckout/payment_checkout')->getCode();

        return $quote && $quote->getPayment()->getMethod() === $method;
    }

    /**
     * @param Mage_Sales_Model_Quote $quote
     * @return bool
     */
    public function quoteIsLockedForChanges(Mage_Sales_Model_Quote $quote)
    {
        $status = $quote->getPayment()->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_INVOICE_STATUS);

        return Crossroads_Collector_Helper_Data::INVOICE_STATUS_PRELIMINARY === $status;
    }
}
