<?php

class Crossroads_CategoryFields_Model_Observer extends Mage_Core_Model_Abstract
{
    /**
     * Constant array for keys to use to find images for the category to add to the
     * prepared category data.
     */
    protected $_imageKeys = [
        "small"  => "cls_catfields_general_img_s",
        "medium" => "cls_catfields_general_img_m",
        "large"  => "cls_catfields_general_img_l"
    ];

    public function __construct()
    {
        $this->category_image_path = Mage::getBaseDir('media') . DS . 'catalog' . DS . 'category' . DS;
        $this->category_prefix = "cls_catfields_general_img_";
        $this->min_size = 50;
    }

    protected function getTempImages($images) {
        if(empty($images)) {
            return [];
        }

        $db  = Mage::getSingleton('core/resource')->getConnection('core_read');

        $sql = "SELECT `name`, `data` FROM `crossroads_categoryimagecache`
                WHERE name IN (" . implode(",", array_fill(0, count($images), "?")) . ")";

        return $db->query($sql, array_values($images))->fetchAll(Zend_Db::FETCH_ASSOC);
    }

    protected static function getImageData($images, $key) {
        if (!$images) {
            return null;
        }

        foreach($images as $d) {
            if($d["name"] === $key) {
                return $d["data"];
            }
        }

        return null;
    }

    /**
     * Retrieves category images map with keys being the image type and value being a map
     * with url being absolute url to the image and dataUrl being a dataUrl which can be
     * used as a placeholder.
     *
     * @return Associative array
     */
    protected function getCategoryImages($category) {
        $imageNames = array_filter(array_map(function($k) use($category) {
            return $category->getData($k);
        }, $this->_imageKeys));

        $tempImages = $this->getTempImages($imageNames);

        return array_map(function($name) use($tempImages) {
            return [
                "url"     => Mage::getBaseUrl("media")."catalog/category/".$name,
                "dataUrl" => self::getImageData($tempImages, $name),
            ];
        }, $imageNames);
    }

    public function prepareApiCategoryData(Varien_Event_Observer $observer) {
        $category      = $observer->getCategory();
        $data          = $observer->getPreparedData();
        $includeImages = $observer->getIncludeImages();

        // Use stdClass to enforce object if empty
        $images = $includeImages ? $this->getCategoryImages($category) : new stdClass();

        $data->setData("images", $images);
        $data->setData("hero", [
            "heading"     => $category->getClsCatfieldsHeroHeading(),
            "subheading"  => $category->getClsCatfieldsHeroSubHeading(),
            "description" => $category->getClsCatfieldsHeroDesc(),
            "offerIcon"   => $category->getClsCatfieldsHeroPercentage(),
            "modifier"    => $category->getClsCatfieldsHeroModifier(),
            "button" => [
                "text" => $category->getClsCatfieldsHeroCtaText(),
                "link" => $category->getClsCatfieldsHeroLink()
            ]
        ]);
    }

    protected function gcd($a,$b) {
        return ($a % $b) ? $this->gcd($b,$a % $b) : $b;
    }

    /**
     * @param $path
     * @return int width
     */
    protected function getSmallestPossibleImage($_file_name) {
        $_media_dir = Mage::getBaseDir('media') . DS . 'catalog' . DS . 'category' . DS;
        $img        = new Varien_Image($_media_dir . $_file_name);

        $w = $img->getOriginalWidth();
        $h = $img->getOriginalHeight();

        $newWidth = $w / $this->gcd($w, $h);

        if($newWidth < $this->min_size) {
            return $newWidth * max(1, ceil($this->min_size / $newWidth));
        }

        return $newWidth;
    }

    protected function resizeCategoryImage($_file_name, $width) {
        $_media_dir = Mage::getBaseDir('media') . DS . 'catalog' . DS . 'category' . DS;
        $cache_dir  = $_media_dir . 'cache' . DS;

        if (file_exists($_media_dir . $_file_name)) {
            if (!is_dir($cache_dir)) {
                mkdir($cache_dir);
            }

            $_image = new Varien_Image($_media_dir . $_file_name);

            $_image->constrainOnly(true);
            $_image->keepAspectRatio(true);
            $_image->keepTransparency(true);
            $_image->resize($width);
            $_image->save($cache_dir . $_file_name);

            return Mage::getBaseUrl('media') . DS . 'catalog' . DS . 'category' . DS . 'cache' . DS . $_file_name;
        }
    }

    protected function getImageDataUri($path) {
        if (!file_get_contents($path)) {
            return null;
        }

        $filename        = substr($path, strrpos($path, '/') + 1);
        $width           = $this->getSmallestPossibleImage($filename);
        $image_small_gcd = $this->resizeCategoryImage($filename, $width);
        $img_file        = file_get_contents($image_small_gcd);

        if (!$img_file) {
            return null;
        }

        $meta = getimagesize($image_small_gcd);

        return "data:" . $meta["mime"] . ";base64," . base64_encode($img_file);
    }

    public function saveCategoryData(Varien_Event_Observer $observer)
    {
        $db_write           = Mage::getSingleton('core/resource')->getConnection('core_write');
        $db_read            = Mage::getSingleton('core/resource')->getConnection('core_read');
        $categoryDataObject = $observer->getEvent()->getDataObject();
        $categoryData       = $categoryDataObject->getData();
        $images             = [];

        foreach ($categoryData as $key => $field) {
            if (strpos($key, $this->category_prefix) === 0) {
                $v        = is_array($field) ? $field["value"] : $field;
                $dataURI  = $this->getImageDataUri($this->category_image_path . $v);
                $image_id = $db_read->query("SELECT value_id FROM catalog_category_entity_varchar WHERE value = ?", [$v])->fetchColumn();

                $images[$v] = [
                    "data"     => $dataURI,
                    "image_id" => $image_id
                ];
            }
        }

        foreach ($images as $name => $image) {
            if (empty($image["data"])) {
                continue;
            }

            $query = 'INSERT INTO crossroads_categoryimagecache (`name`, `data`) VALUES (?, ?)
                      ON DUPLICATE KEY UPDATE `data` = ?;';
            $db_write->query($query, [$name, $image["data"], $image["data"]]);
        }

    }
}
