<?php

abstract class Crossroads_API_Payment_Abstract {
    const ORDER_STATUS_COMPLETE = "COMPLETE";
    const ORDER_IN_PROGRESS     = "IN_PROGRESS";
    const ORDER_FAILED          = "FAILED";

    /**
     * Returns the payment method code associated with this model. This is
     * also used as the third segment in paths to payment-specific controllers,
     * eg. `/api/payment/my_payment_code/my_payment_action`.
     *
     * @return string
     */
    public abstract function getCode();

    /**
     * Returns the title of the payment method. Default implementation reads from
     * `payment/{$this->getCode()}/title`.
     *
     * @return string
     */
    public function getTitle() {
        return Mage::getStoreConfig("payment/" . $this->getCode() . "/title");
    }

    /**
     * If the payment method has a specific controller providing extra actions for checkout or
     * post checkout actions this method returns a string with the controller name. This controller
     * will be instantiated using Mage::getControllerInstance().
     *
     * @return string|null
     */
    public function getControllerClassName() {
        return null;
    }

    /**
     * Returns payment-specific data for the given quote. Presented under `paymentMethodData` in
     * `GET /checkout` and `PUT /checkout`.
     *
     * NOTE: The key `method` is reserved for use by the system. Payment methods are forbidden from
     *       using this key in the `paymentMethodData`.
     *
     * NOTE: The `paymentMethodData` data will be passed to `$payment->importData`, passing all its
     *       parameters to the `assignData` method of the payment type.
     *
     * @param  Mage_Sales_Model_Quote
     */
    public function getPaymentMethodData($quote) {
        return null;
    }

    /**
     * Returns one of the ORDER_* constants from this class depending on the order's payment status.
     */
    public function getOrderPaymentStatus($order) {
        switch($order->getStatus()) {
        case Mage_Sales_Model_Order::STATE_PENDING_PAYMENT:
        case Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW:
            return self::ORDER_IN_PROGRESS;
        default:
            return self::ORDER_STATUS_COMPLETE;
        }
    }

    /**
     * Return falseish if the payment data is valid, if the response is trueish it will be
     * supplied as the value of the data key in a 400 response.
     *
     * @param  Mage_Sales_Model_Quote
     */
    public abstract function isQuoteInvalid($quote);

    /**
     * Returns data for progressing payment. This method should match the `getOrderPaymentStatus`,
     * if it is ORDER_COMPLETE it should be null.
     */
    public abstract function getOrderPaymentData($order);

    /**
     * Returns the root-relative URI to the controller returned by getControllerClassName(),
     * the URI does not contain any preceeding or tailing slashes.
     *
     * @return string
     */
    public function getControllerUrl() {
        return trim(Crossroads_API_Helper_Payment::BASE_URI . "/" . $this->getCode(), "/");
    }
}
