<?php

class Crossroads_API_Helper_Product {
    /**
     * Executed after the basic product associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_POST_DATA_PREPARE      = "crossroads_api_product_post_data_prepare";

    /**
     * Executed after the basic product-list associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_LIST_POST_DATA_PREPARE = "crossroads_api_product_list_post_data_prepare";

    /**
     * Executed after the basic product-cart associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * item           The cart item
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_CART_POST_DATA_PREPARE = "crossroads_api_product_cart_post_data_prepare";

    const PRODUCT_CONFIGURABLE = "configurable";
    const PRODUCT_BUNDLE       = "bundle";

    /**
     * Image-size, smallet dimension, pixels.
     *
     * @var number
     */
    protected $smallImageSize;
    /**
     * Image-size, large dimension, pixels.
     *
     * @var number
     */
    protected $largeImageSize;
    /**
     * Image-size, thumbnail dimension, pixels.
     *
     * @var number
     */
    protected $thumbImageSize;
    protected $smallImageFill;
    protected $largeImageFill;
    protected $thumbImageFill;
    /**
     * If to include stock quantity in product detail, cart and cart options.
     *
     * @var boolean
     */
    protected $getStockQty        = false;
    /**
     * If to include stock quantity in product listings.
     *
     * @var boolean
     */
    protected $getStockQtyList    = false;
    /**
     * If to include media gallery in product detail.
     *
     * @var boolean
     */
    protected $getMediaGallery    = false;
    /**
     * If to include related products in product detail.
     *
     * @var boolean
     */
    protected $getRelatedProducts = false;
    /**
     * "Constant" list of product-types which has options.
     *
     * @var array
     */
    protected $TYPES_WITH_OPTIONS = ["configurable", "bundle"];

    protected $productAttributesView = null;
    protected $productAttributesList = null;
    protected $productAttributesCart = null;
    protected $optionAttributes = null;
    /**
     * Customer group id => customer group name map.
     *
     * @var array
     */
    protected $customerGroups = [];

    /**
     * Loads store-specific configuration for product details and listings.
     */
    public function __construct()
    {
        $store = Mage::app()->getStore();

        $this->largeImageSize        = $store->getConfig('API_section/images/large_image_size');
        $this->largeImageFill        = $store->getConfig('API_section/images/large_image_fill');
        $this->smallImageSize        = $store->getConfig('API_section/images/small_image_size');
        $this->smallImageFill        = $store->getConfig('API_section/images/small_image_fill');
        $this->thumbImageSize        = $store->getConfig('API_section/images/thumbnail_image_size');
        $this->thumbImageFill        = $store->getConfig('API_section/images/thumbnail_image_fill');
        $this->getStockQty           = $store->getConfig('API_section/product_view/get_stock_qty');
        $this->getMediaGallery       = $store->getConfig('API_section/product_view/get_media_gallery');
        $this->getRelatedProducts    = $store->getConfig('API_section/product_view/get_related_products');
        $this->getGroupPrices        = $store->getConfig('API_section/product_view/get_group_prices');
        $this->getStockQtyList       = $store->getConfig('API_section/product_list/get_stock_qty_in_list');
        $this->getGroupPricesInList  = $store->getConfig('API_section/product_list/get_group_prices_in_list');
        $this->getOptionsInList      = $store->getConfig('API_section/product_list/get_options_in_list');
        $this->includeThumbnail      = $store->getConfig('API_section/product_list/include_thumbnail');
        $this->productAttributesView = unserialize($store->getConfig('API_section/attributes/product_view'));
        $this->productAttributesList = unserialize($store->getConfig('API_section/attributes/product_list'));
        $this->productAttributesCart = unserialize($store->getConfig('API_section/attributes/product_cart'));
        $this->optionAttributes      = unserialize($store->getConfig('API_section/attributes/product_option'));

        if($this->getGroupPrices || $this->getGroupPricesInList) {
            $this->customerGroups = $this->loadCustomerGroupMap();
        }
    }

    protected function loadCustomerGroupMap() {
        return array_map(function($group) {
            return $group->getCustomerGroupCode();
        }, Mage::getModel("customer/group")->getCollection()->getItems());
    }

    /**
     * Prepares a product for viewing in a category-listing, prepared for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareListProduct($product)
    {
        $thumbnail_image = null;
        $msrp            = $this->getMsrp($product);

        $small_image = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->keepFrame($this->smallImageFill)
            ->resize($this->smallImageSize);

        if ($this->includeThumbnail) {
            $thumbnail_image = (string)Mage::helper("catalog/image")
                ->init($product, 'small_image')
                ->keepFrame($this->thumbImageFill)
                ->resize($this->thumbImageSize);
        }

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$product->getMinimalPrice() / $msrp)) * 100)
            : null;

        $productData = new Varien_Object(array_merge([
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),
            "urlKey"           => $product->getUrlKey(),

            "price"            => (double)$product->getMinimalPrice(),
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            // We need to manually fetch the group-price attribute since they are not included for a list product
            "groupPrices"      => $this->getGroupPricesInList ? $this->prepareGroupPrices(Mage::getResourceSingleton("catalog/product_attribute_backend_groupprice")->loadPriceData($product->getId(), Mage::app()->getStore()->getId())) : null,
            "originalPrice"    => (double)$product->getPrice(),
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "shortDescription" => $product->getShortDescription(),
            "thumbImage"       => $thumbnail_image,
            "smallImage"       => $small_image,
            "isSalable"        => $product->getIsSalable() > 0,
            "position"         => $product->getPosition(),
            "options"          => $this->getOptionsInList && $product->getTypeId() === self::PRODUCT_CONFIGURABLE ?
                $this->prepareConfigurableOptions($product, false) : null,
            "attributes"       => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesList)
        ], $this->getStockInfo($product, $this->getStockQtyList)));

        Mage::dispatchEvent(self::EVENT_LIST_POST_DATA_PREPARE, [
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Converts a Product into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareProductDetail($product) {
        list($price, $originalPrice) = $product->getTypeId() === self::PRODUCT_BUNDLE ?
            $product->getPriceModel()->getTotalPrices($product, null, true) :
            [$product->getFinalPrice(), $product->getPrice()];
        $small_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->keepFrame($this->smallImageFill)
            ->resize($this->smallImageSize);
        $large_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'image')
            ->keepFrame($this->largeImageFill)
            ->resize($this->largeImageSize);
        $original_image  = (string)Mage::helper("catalog/image")
            ->keepFrame($this->thumbImageFill)
            ->init($product, 'image');
        $msrp            = $this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$price / $msrp)) * 100)
            : null;

        $inStock = $product->getIsInStock() > 0;

        // when the configurable products children is out of stock the price
        // gets calculated to "0". To avoid loading each child we simply assume
        // the product is out of stock.
        if ($product->getTypeId() === self::PRODUCT_CONFIGURABLE && !$price) {
            $inStock = false;
        }

        $productData = new Varien_Object(array_merge([
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),

            "price"            => (double)$price,
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            // Included normally when a product is ->load()ed
            "groupPrices"      => $this->getGroupPrices ? $this->prepareGroupPrices($product->getData("group_price") ?: []) : null,
            "categoryIds"      => array_map(function($id) { return (int)$id; }, $product->getCategoryIds() ?: []),
            "originalPrice"    => (double)$originalPrice,
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "metaDescription"  => $product->getMetaDescription(),
            "shortDescription" => $product->getShortDescription(),
            "description"      => $product->getDescription(),
            "urlKey"           => $product->getUrlKey(),
            "smallImage"       => $small_image,
            "largeImage"       => $large_image,
            "originalImage"    => $original_image,
            "isInStock"        => $inStock,
            "isSalable"        => $product->getIsSalable() > 0,
            "options"          => $product->getTypeId() === self::PRODUCT_CONFIGURABLE ?
                    $this->prepareConfigurableOptions($product, true) : null,
            "selections"       => $product->getTypeId() === self::PRODUCT_BUNDLE ?
                $this->prepareBundleOptions($product, true) : null,
            "relatedProducts"  => $this->getRelatedProducts ? $this->prepareRelatedProducts($product) : null,
            "mediaGallery"     => $this->getMediaGallery ? $this->prepareMediaGallery($product) : null,
            "customOptions"    => $this->prepareCustomOptions($product),
            "attributes"       => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesView),
        ], $this->getStockInfo($product, $this->getStockQty)));

        Mage::dispatchEvent(self::EVENT_POST_DATA_PREPARE, [
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Converts a Cart item into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product cart data
     */
    public function prepareCartProduct($item)
    {
        $product         = $item->getProduct();
        $rowTotal        = $item->getRowTotal()
            + $item->getTaxAmount()
            + $item->getHiddenTaxAmount()
            - $item->getDiscountAmount();

        $thumbnail = (string)Mage::helper("catalog/image")
            ->init($product, 'thumbnail')
            ->keepFrame($this->thumbImageFill)
            ->resize($this->thumbImageSize);

        $attributes    = null;
        $bundleOptions = null;
        $customOptions = null;
        $options       = $product->getTypeInstance(true)->getOrderOptions($product);

        if( ! empty($options["info_buyRequest"]) &&
            is_array($options["info_buyRequest"])) {
            if(array_key_exists("super_attribute", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["super_attribute"])) {
                $attributes = [];

                foreach($options["info_buyRequest"]["super_attribute"] as $k => $v) {
                    $attributes["$k"] = (int)$v;
                }
            }

            if(array_key_exists("bundle_option", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["bundle_option"])) {
                $bundleOptions = [];

                foreach($options["info_buyRequest"]["bundle_option"] as $k => $v) {
                    $bundleOptions["$k"] = is_array($v) ? array_values(array_map(function($i) { return (int)$i; }, $v)): (int)$v;
                }
            }

            if(array_key_exists("options", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["options"])) {
                $customOptions = [];

                foreach($options["info_buyRequest"]["options"] as $k => $v) {
                    $customOptions["$k"] = $v;
                }
            }
        }

        $productData = new Varien_Object([
            "id"        => (int)$item->getId(),
            "product"   => array_merge([
                "id"              => (int)$product->getId(),
                "sku"             => $product->getSku(),
                "name"            => $product->getName(),
                "urlKey"          => $product->getUrlPath(),
                "price"           => (double)$item->getPriceInclTax(),
                "isSalable"       => $product->getIsSalable() > 0,
                "thumbnail"       => $thumbnail,
                "attributes"      => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesCart)
            ], $this->getStockInfo($item->getOptionByCode('simple_product') ? $item->getOptionByCode("simple_product")->getProduct() : $product, $this->getStockQty)),
            "qty"           => (double)($item->getQty() ?: $item->getQtyOrdered()),
            "rowTotal"      => (double)$rowTotal,
            "rowTax"        => (double)$item->getTaxAmount(),
            "attributes"    => $attributes,
            "options"       => $product->getTypeId() === self::PRODUCT_CONFIGURABLE ?
                $this->prepareOptionsCart($product, $options) : null,
            "customOptions" => $customOptions,
            "bundleOptions" => $bundleOptions,
            "selections"    => $product->getTypeId() === self::PRODUCT_BUNDLE ?
                $this->prepareSelectionsCart($product, $options) : null,
        ]);

        Mage::dispatchEvent(self::EVENT_CART_POST_DATA_PREPARE, [
            "item"          => $item,
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareOptionsCart($product, $options) {
        if (empty($options["info_buyRequest"]) ||
            !is_array($options["info_buyRequest"]["super_attribute"])) {
            return null;
        }

        $helper     = $this;
        $attrs      = $options["info_buyRequest"]["super_attribute"];
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = $instance->getUsedProducts(null, $product);

        // Specific options with specific set values
        return array_map(function($option) use($attrs, $children, $helper) {
            $value = null;
            $child = null;

            // Attempt to find the specific child-product for the selected attribute value
            foreach($children as $c) {
                if((int)$c->getData($option["attribute_code"]) === (int)$attrs[$option["attribute_id"]]) {
                    $child = $c;

                    break;
                }
            }

            // Find the selected attribute-value
            foreach($option["values"] as $v) {
                if((int)$attrs[$option["attribute_id"]] === (int)$v["value_index"]) {
                    $value = $v;

                    break;
                }
            }

            if($value && ! $child) {
                throw new Exception(sprintf("Cart attribute %s: %s failed to obtain a child product.", $option["attribute_code"], $v["value_index"]));
            }

            $thumbnail = $child->getThumbnail() && $child->getThumbnail() !== "no_selection" ? (string)Mage::helper("catalog/image")
                ->init($child, 'thumbnail')
                ->keepFrame($this->thumbImageFill)
                ->resize($this->thumbImageSize) : null;

            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                // There might be a case where we cannot find the selected value
                "value"        => $value ? array_merge([
                    "id"         => (int)$value["value_index"],
                    "sku"        => $child->getSku(),
                    "label"      => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                    "isPercent"  => $value["is_percent"] > 0,
                    "isInStock"  => $child->isInStock() > 0,
                    "isSalable"  => $child->getIsSalable() > 0,
                    "thumbnail"  => $thumbnail,
                    "price"      => (double)$child->getPrice(),
                    "msrp"       => $child->getMsrp() ? (double)$child->getMsrp() : null,
                    "attributes" => Mage::helper("API/attributes")->getEntityAttributes($child, $helper->optionAttributes)
                ], $this->getStockInfo($child, $this->getStockQty)) : null
            ];
        }, $rawOptions);
    }

    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareSelectionsCart($product, $options) {
        if (empty($options["info_buyRequest"]) ||
            !is_array($options["bundle_options"])) {
            return null;
        }

        $selections = [];

        foreach($options["bundle_options"] as $selectionId => $selection) {
            $selections[] = [
                "id"      => (int)$selectionId,
                "title"   => $selection["label"],
                "options" => array_values(array_map(function($productData) {
                    return [
                        "name"  => $productData["title"],
                        "qty"   => (int)$productData["qty"],
                        "price" => (double)$productData["price"],
                    ];
                }, $selection["value"])),
            ];
        }

        return $selections;
    }


    protected function prepareGroupPrices($groupPrices) {
        $customerGroups = $this->customerGroups;
        $websiteId      = Mage::app()->getStore()->getWebsiteId();

        return array_values(array_filter(array_map(function($price) use($customerGroups, $websiteId) {
            if($price["website_id"] != $websiteId && $price["website_id"] != 0) {
                return null;
            }

            return [
                "groupCode" => array_key_exists($price["cust_group"], $customerGroups) ? $customerGroups[$price["cust_group"]] : null,
                "price"     => (double)$price["price"]
            ];
        }, $groupPrices)));
    }

    /**
     * Obtains the MSRP for a given product, if it is a complex product will will find the smallest MSRP
     * of the child products.
     *
     * @param  Mage_Catalog_Model_Product
     * @return double
     */
    protected function getMsrp(Mage_Catalog_Model_Product $product) {
        if ($product->getTypeId() !== self::PRODUCT_CONFIGURABLE) {
            return (double)$product->getMsrp();
        }

        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        $msrps = [];

        foreach($rawOptions as $option) {
            foreach($children as $child) {
                foreach ($option["values"] as $value)  {
                    if ($child->getData($option["attribute_code"]) === $value["value_index"]) {
                        $msrp = $child->getMsrp();
                        if ($msrp) {
                            $msrps[] = $msrp;
                        }
                    }
                }
            }
        }

        return !empty($msrps) ? (double)min($msrps) : null;
    }

    /**
     * Prepares product option-list for a complex product.
     *
     * @param  Mage_Catalog_Model_Product  Configurable product
     * @param  boolean     If to include stock for each variant
     * @return array
     */
    protected function prepareConfigurableOptions($product, $includeStock) {
        $helper     = $this;
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        return array_map(function($option) use($children, $helper, $includeStock) {
            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                "values"       => array_values(array_filter(array_map(function($value) use($children, $option, $helper, $includeStock) {
                    // Find the correct child product and output its data
                    foreach($children as $child) {
                        if($child->getData($option["attribute_code"]) !== $value["value_index"]) {
                            continue;
                        }

                        $small_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'small_image')
                            ->keepFrame($this->smallImageFill)
                            ->resize($this->smallImageSize);
                        $large_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'image')
                            ->keepFrame($this->largeImageFill)
                            ->resize($this->largeImageSize);
                        $original_image = (string)Mage::helper("catalog/image")
                            ->init($child, 'image');

                        // TODO: Maybe skip this in lists anyway?
                        if($this->getGroupPrices) {
                            $attribute = $child->getResource()->getAttribute("group_price");

                            if($attribute) {
                                $attribute->getBackend()->afterLoad($child);
                            }
                        }

                        return array_merge([
                            "id"            => (int)$value["value_index"],
                            "sku"           => $child->getSku(),
                            "smallImage"    => $small_image,
                            "largeImage"    => $large_image,
                            "originalImage" => $original_image,
                            "label"         => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                            "isPercent"     => $value["is_percent"] > 0,
                            "isInStock"     => $child->isInStock() > 0,
                            "isSalable"     => $child->getIsSalable() > 0,
                            "groupPrices"   => $this->getGroupPrices ? $this->prepareGroupPrices($child->getData("group_price") ?: []) : null,
                            "specialPrice"  => $child->getSpecialPrice() > 0 ? ((double)$child->getSpecialPrice()) : null,
                            "price"         => (double)$child->getFinalPrice(),
                            "originalPrice" => (double)$child->getPrice(),
                            "msrp"          => $child->getMsrp() ? (double)$child->getMsrp() : null,
                            "customOptions" => $this->prepareCustomOptions($child),
                            "attributes"    => Mage::helper("API/attributes")->getEntityAttributes($child, $helper->optionAttributes)
                        ], $this->getStockInfo($child, $includeStock && $this->getStockQty));
                    }

                    // Empty, array_filter above will remove empty entries
                    return null;
                }, $option["values"])))
            ];
        }, $rawOptions);
    }

    protected function prepareBundleOptions($product, $includeStock) {
        $instance = $product->getTypeInstance();

        return array_values(array_map(function($option) use($includeStock, $product) {
            return [
                "id"       => (int)$option->getOptionId(),
                "required" => (boolean)$option->getRequired(),
                "position" => (int)$option->getPosition(),
                "type"     => $option->getType(),
                "title"    => $option->getTitle() ?: $option->getDefaultTitle(),
                "options"  => array_map(function($selection) use($includeStock, $product) {
                    $small_image     = (string)Mage::helper("catalog/image")
                        ->init($selection, 'small_image')
                        ->keepFrame($this->smallImageFill)
                        ->resize($this->smallImageSize);
                    $large_image     = (string)Mage::helper("catalog/image")
                        ->init($selection, 'image')
                        ->keepFrame($this->largeImageFill)
                        ->resize($this->largeImageSize);
                    $original_image = (string)Mage::helper("catalog/image")
                        ->init($selection, 'image');

                    if($this->getGroupPrices) {
                        $attribute = $selection->getResource()->getAttribute("group_price");

                        if($attribute) {
                            $attribute->getBackend()->afterLoad($selection);
                        }
                    }

                    $price = 0.0;

                    switch($selection->getSelectionPriceType()) {
                    case 0: // Fixed
                        $price = $selection->getSelectionPriceValue();
                        break;
                    case 1: // Percent
                        $price = $product->getFinalPrice() * ($selection->getSelectionPriceValue() / 100);
                        break;
                    }

                    return array_merge([
                        "id"               => (int)$selection->getSelectionId(),
                        "isDefault"        => (bool)$selection->getIsDefault(),
                        "qty"              => (int)$selection->getSelectionQty(),
                        "isQtyFixed"       => !$selection->getSelectionCanChangeQty(),
                        "name"             => $selection->getName(),
                        "shortDescription" => $selection->getShortDescription(),
                        "smallImage"       => $small_image,
                        "largeImage"       => $large_image,
                        "originalImage"    => $original_image,
                        "isInStock"        => $selection->isInStock() > 0,
                        "isSalable"        => $selection->getIsSalable() > 0,
                        "groupPrices"      => $this->getGroupPrices ? $this->prepareGroupPrices($selection->getData("group_price") ?: []) : null,
                        "price"            => (double)$price,
                        "msrp"             => $selection->getMsrp() ? (double)$selection->getMsrp() : null,
                        "attributes"       => Mage::helper("API/attributes")->getEntityAttributes($selection, $this->optionAttributes),
                    ], $this->getStockInfo($selection, $includeStock && $this->getStockQty));
                }, $option->getSelections() ?: []),
            ];
        }, $instance->getOptions()));
    }

    protected function prepareMediaGallery($product)
    {
        $coll = $product->getMediaGalleryImages();

        if( ! $coll) {
            return [];
        }

        return array_values(array_map(function($image) use ($product) {
            return [
                "thumbnail" => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->keepFrame($this->thumbImageFill)
                    ->resize($this->thumbImageSize),
                "original"  => Mage::getModel('catalog/product_media_config')
                    ->getMediaUrl($image->getFile()),
                "large"     => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->keepFrame($this->largeImageFill)
                    ->resize($this->largeImageSize),
                "image"     => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->keepFrame($this->smallImageFill)
                    ->resize($this->smallImageSize),
            ];
        }, $coll->getItems()));
    }

    protected function prepareCustomOptions($product) {
        if( ! $product->getHasOptions()) {
            return null;
        }

        $options = array_values($product->getProductOptionsCollection()->getItems());

        if(empty($options)) {
            return null;
        }

        return array_map(function($option) {
            return [
                "id"        => (int)$option->getOptionId(),
                "type"      => $option->getType(),
                // NOT A TYPO
                "required"  => (bool)$option->getIsRequire(),
                "maxLength" => (int)$option->getMaxCharacters(),
                "title"     => $option->getStoreTitle() ?: $option->getTitle() ?: $option->getDefaultTitle(),
                "price"     => (double)$option->getPrice(true),
                "values"    => array_map(function($value) {
                    return [
                        "sku"   => $value->getSku(),
                        "title" => $value->getStoreTitle() ?: $value->getTitle() ?: $value->getDefaultTitle(),
                        "price" => (double)$value->getPrice(true),
                    ];
                }, $option->getValues()),
            ];
        }, $options);
    }

    protected function getStockInfo($product, $loadValues) {
        $stockQty        = null;
        $stockBackorders = null;
        $stockManage     = null;

        if($loadValues) {
            $stock = Mage::getModel('cataloginventory/stock_item')->loadByProduct($product);

            $stockBackorders = (boolean)$stock->getBackorders();
            $stockManage     = (boolean)$stock->getManageStock();
            $stockQty        = $stockManage ? (double)$stock->getQty() : null;
        }

        return [
            "stockQty"        => $stockQty,
            "stockBackorders" => $stockBackorders,
            "stockManage"     => $stockManage,
        ];
    }

    protected function prepareRelatedProducts($product)
    {
        return array_values(array_map([$this, "prepareListProduct"], $product->getRelatedProductCollection()
            ->addAttributeToSelect('*')
            ->setPositionOrder()
            ->addMinimalPrice()
            ->addStoreFilter(Mage::app()->getStore()->getStoreId())
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            )->getItems()));
    }
}
