<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;

require_once("Mage.php");
require_once("Model/Serializer/Abstract.php");
require_once("Model/Serializer/Groupprices.php");

/**
 * @covers Crossroads_API_Model_Serializer_Groupprices
 */
class Crossroads_API_Model_Serializer_GrouppricesTest extends TestCase {
    public function setUp() {
        Mage::_clear();
    }

    public function testGetCustomerGroups() {
        $g1 = $this->getMockBuilder(stdClass::class)
                   ->setMethods(["getCustomerGroupCode"])
                   ->getMock();
        $g2 = $this->getMockBuilder(stdClass::class)
                   ->setMethods(["getCustomerGroupCode"])
                   ->getMock();
        $g1->expects($this->once())
           ->method("getCustomerGroupCode")
           ->will($this->returnValue("Group 1"));
        $g2->expects($this->once())
           ->method("getCustomerGroupCode")
           ->will($this->returnValue("Group 2"));

        $collection = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getItems"])
                           ->getMock();
        $collection->expects($this->once())
                   ->method("getItems")
                   ->will($this->returnValue([
                       4 => $g1,
                       9 => $g2,
                   ]));
        $mock = $this->getMockBuilder(stdClass::class)
                     ->setMethods(["getCollection"])
                     ->getMock();
        $mock->expects($this->once())
             ->method("getCollection")
             ->will($this->returnValue($collection));

        Mage::_initConfig([ "store" => [ ] ]);
        Mage::_setModel("customer/group", $mock);

        $serializer = $this->getMockBuilder(Crossroads_API_Model_Serializer_Groupprices::class)
                           ->setMethods(["serializeItem"])
                           ->getMock();

        $this->assertEquals($serializer->getCustomerGroups(), [
            4 => 'Group 1',
            9 => 'Group 2'
        ]);
        // Twice to make sure it is stored
        $this->assertEquals($serializer->getCustomerGroups(), [
            4 => 'Group 1',
            9 => 'Group 2'
        ]);
    }

    /**
     * @dataProvider serializeGroupPricesAttributes
     */
    public function testSerializeGroupPrices($websiteId, $customerGroups, $prices, $expected) {
        Mage::_initConfig([ "store" => [ "websiteId" => $websiteId ] ]);

        $groups = $this->getMockBuilder(stdClass::class)
                       ->setMethods(["getCollection"])
                       ->getMock();
        $collection = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getItems"])
                           ->getMock();
        $collection->expects($this->once())
                   ->method("getItems")
                   ->will($this->returnValue(array_map(function($g) {
            $go = $this->getMockBuilder(stdClass::class)
                       ->setMethods(["getCustomerGroupCode"])
                       ->getMock();
            $go->expects($this->once())
               ->method("getCustomerGroupCode")
               ->will($this->returnValue($g));

            return $go;
                   }, $customerGroups)));
        $groups->expects($this->once())
               ->method("getCollection")
               ->will($this->returnValue($collection));

        Mage::_setModel("customer/group", $groups);

        $serializer = new Crossroads_API_Model_Serializer_Groupprices([
            "websiteId" => $websiteId,
        ]);

        $this->assertSame($expected, $serializer->serializeItem($prices));
    }

    public function serializeGroupPricesAttributes() {
        return [
            [0, [], [], []],
            [0, [], [["website_id" => 0, "cust_group" => 4, "price" => 12]], [["groupCode" => null, "price" => 12.0]]],
            [0, [], [["website_id" => 1, "cust_group" => 4, "price" => 12]], []],
            [1, [], [["website_id" => 0, "cust_group" => 4, "price" => 12]], [["groupCode" => null, "price" => 12.0]]],
            [1, [], [["website_id" => 1, "cust_group" => 4, "price" => 12]], [["groupCode" => null, "price" => 12.0]]],
            [0, [4 => "Group A"], [["website_id" => 0, "cust_group" => 4, "price" => 12]], [["groupCode" => "Group A", "price" => 12.0]]],
            [0, [4 => "Group A"], [["website_id" => 1, "cust_group" => 4, "price" => 12]], []],
            [1, [4 => "Group A"], [["website_id" => 0, "cust_group" => 4, "price" => 12]], [["groupCode" => "Group A", "price" => 12.0]]],
            [1, [4 => "Group A"], [["website_id" => 1, "cust_group" => 4, "price" => 12]], [["groupCode" => "Group A", "price" => 12.0]]],
            [1, [6 => "Group B"], [["website_id" => 1, "cust_group" => 4, "price" => 12]], [["groupCode" => null, "price" => 12.0]]],
            [1, [4 => "Group A", 6 => "Group B"], [["website_id" => 0, "cust_group" => 6, "price" => 17], ["website_id" => 1, "cust_group" => 4, "price" => 12]], [["groupCode" => "Group B", "price" => 17.0], ["groupCode" => "Group A", "price" => 12.0]]],
            [1, [4 => "Group A", 6 => "Group B"], [["website_id" => 1, "cust_group" => 4, "price" => 12], ["website_id" => 0, "cust_group" => 6, "price" => 17]], [["groupCode" => "Group A", "price" => 12.0], ["groupCode" => "Group B", "price" => 17.0]]],
            [1, [4 => "Group A", 6 => "Group B"], [["website_id" => 1, "cust_group" => 4, "price" => 12], ["website_id" => 6, "cust_group" => 6, "price" => 17]], [["groupCode" => "Group A", "price" => 12.0]]],
        ];
    }
}
