<?php

class Crossroads_API_Helper_Cart
{
    /**
     * Executed after the basic quote associated array has been prepared. This event allows
     * modification of the returned quote associated array.
     *
     * Params:
     *  * quote           The quote object
     *  * totals          The quote totals map, from Mage_Sales_Model_Quote::getTotals()
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     */
    const EVENT_QUOTE_POST_DATA_PREPARE = "crossroads_api_quote_post_data_prepare";

    /**
     * Executed after the basic order associated array has been prepared. This event allows
     * modification of the returned order associated array.
     *
     * Params:
     *  * order           The order object
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     */
    const EVENT_ORDER_POST_DATA_PREPARE = "crossroads_api_order_post_data_prepare";

    /**
     * Executed before Mage_Sales_Model_Quote::addProduct() is called. This event only allows
     * for cancelling additions by throwing a `Crossroads_API_ResponseException`.
     *
     * Params:
     *  * quote      The quote object
     *  * product    The product object
     *  * qty        The new quantity
     *  * attributes The supplied attributes
     */
    const EVENT_QUOTE_PRE_ADD_PRODUCT = "crossroads_api_quote_pre_add_product";

    /**
     * Executed after Mage_Sales_Model_Quote::addProduct() is called.
     *
     * Params:
     *  * quote      The quote object
     *  * product    The product object
     *  * qty        The new quantity
     *  * attributes The supplied attributes
     */
    const EVENT_QUOTE_POST_ADD_PRODUCT = "crossroads_api_quote_post_add_product";

    /**
     * Executed before Mage_Sales_Model_Quote::updateItem() is called. This event only allows
     * for cancelling updates by throwing a `Crossroads_API_ResponseException`.
     *
     * Params:
     *  * quote      The quote object
     *  * product    The product object
     *  * qty        The new quantity
     *  * attributes The supplied attributes
     *  * item       The quote item object to be updated
     */
    const EVENT_QUOTE_PRE_UPDATE_ITEM = "crossroads_api_quote_pre_update_item";

    /**
     * Executed after Mage_Sales_Model_Quote::updateItem() is called.
     *
     * Params:
     *  * quote      The quote object
     *  * product    The product object
     *  * qty        The new quantity
     *  * attributes The supplied attributes
     *  * item       The quote item object to be updated
     */
    const EVENT_QUOTE_POST_UPDATE_ITEM = "crossroads_api_quote_post_update_item";

    /**
     * Executed before Mage_Sales_Model_Quote::removeItem() is called.
     *
     * Params:
     *  * quote:  The quote object
     *  * item:   The item object to be deleted
     */
    const EVENT_QUOTE_PRE_REMOVE_ITEM = "crossroads_api_quote_pre_remove_item";

    /**
     * Executed after Mage_Sales_Model_Quote::removeItem() is called.
     *
     * Params:
     *  * quote:  The quote object
     *  * item:   The item object to be deleted
     */
    const EVENT_QUOTE_POST_REMOVE_ITEM = "crossroads_api_quote_post_remove_item";

    /**
     * Returns the coupon code and rule name applied to the given quote, if any.
     *
     * @param Mage_Sales_Model_Quote
     * @return null or associative array with couponCode and ruleName
     */
    public function getCouponData($quote) {
        if( ! $quote->getCouponCode()) {
            return null;
        }

        $coupon = Mage::getModel("salesrule/coupon")->load($quote->getCouponCode(), "code");

        if(!$coupon || !$coupon->getRuleId()) {
            return null;
        }

        $rule = Mage::getModel("salesrule/rule")->load($coupon->getRuleId());

        return [
            "couponCode" => $quote->getCouponCode(),
            "ruleName"   => $rule->getName(),
        ];
    }

    /**
     * Formats a quote for JSON output.
     *
     * @param  Mage_Sales_Model_Quote
     * @return array
     */
    public function formatQuote($quote) {
        // TODO: Move to a quote serializer
        $store    = Mage::app()->getStore();
        $itemSer  = Mage::getModel("API/factory")->createCartItemSerializer($store);
        $items    = $quote->getAllVisibleItems();
        $qty      = $quote->getQty() ?: 0;
        $totals   = $quote->getTotals();
        $shipping = $quote->getShippingAddress();

        foreach ($items as $item) {
            $qty += $item->getQty();
        }

        $quoteData = new Varien_Object([
            "items"      => $itemSer->mapArray($items),
            "summary"    => [
                "subTotal"              => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueInclTax() : 0,
                "subTotalExclTax"       => array_key_exists("subtotal", $totals) ? (double) $totals["subtotal"]->getValueExclTax() : 0,
                "grandTotal"            => (double)$quote->getGrandTotal(),
                "grandTotalExclTax"     => (double)$quote->getGrandTotal() - (double)$quote->getShippingAddress()->getTaxAmount(),
                "tax"                   => array_key_exists("tax", $totals) ? (double) $totals["tax"]->getValue() : 0,
                "taxRates"              => $this->getTaxRates($quote),
                "discount"              => array_key_exists("discount", $totals) ? (double) $totals["discount"]->getValue() : 0,
                "shippingAmount"        => (double)$shipping->getShippingInclTax() ?: $shipping->getShippingAmount() ?: 0,
                "shippingAmountExclTax" => (double)$shipping->getShippingAmount(),
                "quoteCurrencyCode"     => $quote->getQuoteCurrencyCode(),
                "qty"                   => $qty,
                "coupon"                => $this->getCouponData($quote),
                "virtual"               => $quote->isVirtual(),
                "hasVirtualItems"       => $quote->hasVirtualItems(),
            ]
        ]);

        Mage::dispatchEvent(self::EVENT_QUOTE_POST_DATA_PREPARE, [
            "quote"         => $quote,
            "totals"        => $totals,
            "prepared_data" => $quoteData
        ]);

        return $quoteData->getData();
    }

    /**
     * Formats an order for JSON output.
     *
     * @param  Mage_Sales_Model_Order
     * @return array
     */
    public function formatOrder($order) {
        // TODO: Move to an order serializer
        $store    = Mage::app()->getStore();
        $itemSer  = Mage::getModel("API/factory")->createCartItemSerializer($store);
        $items    = $order->getAllVisibleItems();
        $qty      = $order->getTotalQtyOrdered() ?: 0;
        $shipping = $order->getShippingAddress();

        $orderData = new Varien_Object([
            "items"      => $itemSer->mapArray($items),
            "summary"    => [
                "subTotal"              => (double)$order->getSubtotalInclTax(),
                "subTotalExclTax"       => (double)$order->getSubtotal(),
                "grandTotal"            => (double)$order->getGrandTotal(),
                "grandTotalExclTax"     => (double)$order->getGrandTotal() - (double)$order->getTaxAmount(),
                "tax"                   => (double)$order->getTaxAmount(),
                "taxRates"              => $this->getTaxRates($order),
                "discount"              => (double)$order->getDiscountAmount(),
                "shippingAmount"        => $shipping ? (double)$shipping->getShippingInclTax() ?: $shipping->getShippingAmount() ?: 0 : 0,
                "shippingAmountExclTax" => $shipping ? (double)$shipping->getShippingAmount() : 0,
                "quoteCurrencyCode"     => $order->getOrderCurrencyCode(),
                "qty"                   => (double)$qty,
                "coupon"                => $this->getCouponData($order),
                "virtual"               => (boolean)$order->getIsVirtual(),
                "hasVirtualItems"       => count(array_filter($items, function($i) { return $i->getIsVirtual(); })) > 0,
            ]
        ]);

        Mage::dispatchEvent(self::EVENT_ORDER_POST_DATA_PREPARE, [
            "order"         => $order,
            "prepared_data" => $orderData
        ]);

        return $orderData->getData();
    }

    /**
     * Retrieves a list of the tax rates used in the given quote.
     *
     * @param Mage_Sales_Model_Quote
     * @return array  Array with percent and amount keys for each tax-rate
     */
    protected function getTaxRates($quote) {
        // No idea why it is on shipping address
        $addr  = $quote->getShippingAddress() ?: $quote->getBillingAddress();
        $taxes = $addr->getAppliedTaxes();

        if( ! $taxes) {
            $taxes = method_exists($quote, "getFullTaxInfo") ?  $quote->getFullTaxInfo() : [];
        }

        return array_reduce($taxes, function($taxes, $tax) {
            for($i = 0,$c = count($taxes); $i < $c; $i++) {
                if($taxes[$i]["percent"] === (double)$tax["percent"]) {
                    $taxes[$i]["amount"] += $tax["amount"];

                    return $taxes;
                }
            }

            $taxes[] = [
                "percent" => (double)$tax["percent"],
                "amount"  => (double)$tax["amount"]
            ];

            return $taxes;
        }, []);
    }

    /**
     * Adds and removes multiple items in the cart so that the cart matches the supplied array.
     *
     * NOTE: Does not save the cart
     *
     * @param  Mage_Sales_Model_Quote
     * @param  array  Array of cart items, if no row-id is present it is added, otherwise updated
     */
    public function setCartItems($quote, $items) {
        // We need to delete items
        foreach($quote->getAllVisibleItems() as $cartItem) {
            $exists = false;

            foreach($items as $item) {
                if(array_key_exists("id", $item) && (int)$item["id"] === (int)$cartItem->getItemId()) {
                    $exists = true;

                    break;
                }
            }

            if(!$exists) {
                $this->removeItem($quote, $cartItem);
            }
        }

        // Multiple product requests
        foreach($items as $item) {
            $this->updateItem($quote, $item);
        }
    }

    /**
     * Removes a quote item from a quote.
     *
     * @param  Mage_Sales_Model_Quote
     * @param  Mage_Sales_Model_Quote_Item
     */
    public function removeItem($quote, $item) {
        Mage::dispatchEvent(self::EVENT_QUOTE_PRE_REMOVE_ITEM, [
            "quote" => $quote,
            "item"  => $item
        ]);

        $quote->removeItem($item->getItemId());

        Mage::dispatchEvent(self::EVENT_QUOTE_POST_REMOVE_ITEM, [
            "quote" => $quote,
            "item"  => $item
        ]);
    }

    /**
     * Updates or adds an item to the cart, if an id property is found then it is updated.
     *
     * NOTE: Does not save the cart
     *
     * @param  Mage_Sales_Model_Quote
     * @param  array
     */
    public function updateItem($quote, $item)
    {
        $attributes    = null;
        $bundleOptions = null;
        $customOptions = null;

        if(!is_array($item)) {
            throw Crossroads_API_ResponseException::create(400, "item is not an associative array, got '".gettype($item)."'.", null, 2008);
        }

        // map attributes (configurable product)
        // TODO: Unify with configurable, API-breaking change
        if (array_key_exists("attributes", $item) && $item["attributes"] !== null) {
            if(!is_array($item["attributes"])) {
                throw Crossroads_API_ResponseException::create(400, "item.attributes should be a map, got '".gettype($item["attributes"])."'.", null, 2003);
            }

            if( ! empty($item["attributes"])) {
                $attributes = [];

                foreach($item["attributes"] as $k => $v) {
                    $attributes[(int)$k] = (int)$v;
                }
            }
        }

        // TODO: Unify with bundles/configurable, API-breaking change
        if (array_key_exists("customOptions", $item) && $item["customOptions"] !== null) {
            if(!is_array($item["customOptions"])) {
                throw Crossroads_API_ResponseException::create(400, sprintf("item.customOptions should be a map, got '%s'.", gettype($item["customOptions"])), null, 2014);
            }

            if( ! empty($item["customOptions"])) {
                $customOptions = [];

                foreach($item["customOptions"] as $k => $v) {
                    $customOptions[(int)$k] = $v;
                }
            }
        }

        // map selections (bundle product)
        if (array_key_exists("bundleOptions", $item) && $item["bundleOptions"] !== null) {
            if(!is_array($item["bundleOptions"])) {
                throw Crossroads_API_ResponseException::create(400, sprintf("item.bundleOptions should be a map, got '%s'.", gettype($item["bundleOptions"])), null, 2012);
            }

            if( ! empty($item["bundleOptions"])) {
                $bundleOptions = [];

                foreach($item["bundleOptions"] as $k => $v) {
                    // Bundle options are either optionId => optionValue, or optionId => [optionValue]
                    $bundleOptions[(int)$k] = is_array($v) ? array_values(array_map(function($i) { return (int)$i; }, $v)): (int)$v;
                }
            }
        }

        // Null-check required on $item
        if (empty($item) || empty($item['qty']) || empty($item['product'])) {
            throw Crossroads_API_ResponseException::create(400, "item is missing qty and/or product parameters.", null, 2004);
        }

        if( ! is_array($item["product"]) || !array_key_exists("id", $item["product"])) {
            throw Crossroads_API_ResponseException::create(400, "item is missing product key, or product key does not contain an object with an id property.", null, 2005);
        }

        $product = Mage::getModel('catalog/product')
            ->setStoreId(Mage::app()->getStore()->getId())
            ->load((int)$item["product"]["id"]);

        if (!$product || !$product->getId()) {
            throw Crossroads_API_ResponseException::create(404, "Product not found.", null, 2006);
        }

        if ($product->getTypeId() === "bundle" && empty($bundleOptions)) {
            throw Crossroads_API_ResponseException::create(400, "Bundle product requires options", null, 2013);
        }

        if(!empty($item["id"])) {
            $this->updateProductInCart($quote, (int)$item["id"], $product, $item["qty"], $attributes, $bundleOptions, $customOptions);
        }
        else {
            $this->addProductToCart($quote, $product, $item["qty"], $attributes, $bundleOptions, $customOptions);
        }
    }

    /**
     * Updates a product in the cart.
     *
     * @param  Mage_Sales_Model_Quote
     * @param  int      The row id of the cart row
     * @param  Product  The product to update
     * @param  int      The new quantity
     * @param  mixed    Any product attributes (super_attributes)
     * @param  mixed    Any bundle selection options (bundle_option)
     * @param  mixed    Any custom options (options)
     */
    protected function updateProductInCart($quote, $rowId, $product, $qty, $attributes, $bundleOptions, $customOptions) {
        $request  = Mage::app()->getRequest();
        $response = Mage::app()->getResponse();
        $item     = $quote->getItemById($rowId);

        if( ! $item) {
            throw Crossroads_API_ResponseException::create(400, "Cart item with id '$rowId' was not found when attempting to update cart.", null, 2007);
        }

        Mage::dispatchEvent(self::EVENT_QUOTE_PRE_UPDATE_ITEM, [
            "quote"         => $quote,
            "product"       => $product,
            "qty"           => $qty,
            "attributes"    => $attributes,
            "bundleOptions" => $bundleOptions,
            "customOptions" => $customOptions,
            "item"          => $item
        ]);

        try {
            $quote->updateItem($rowId, new Varien_Object([
                "product"         => $product->getId(),
                "qty"             => $this->clampProductQty($quote, $product, $qty),
                "super_attribute" => $attributes,
                "bundle_option"   => $bundleOptions,
                "options"         => $customOptions,
                // Needed to prevent magento from attempting to fetch the original qty + the new qty
                // if we do not have id here the old quote item is considered different and will be
                // first summed with this request and then deleted after the validation has been performed
                // It also needs to be a string to pass === in Mage_Sales_Model_Quote
                // If it is a different item,
                "id"              => (string)$rowId
            ]));
        }
        catch(Exception $e) {
            // Determnine if we return an error to the user or rethrow
            $this->handleProductException($e, $product);
        }

        // Used for crosssell
        // $this->getCheckoutSession()->setLastAddedProductId($productId);

        Mage::dispatchEvent(self::EVENT_QUOTE_POST_UPDATE_ITEM, [
            "quote"         => $quote,
            "product"       => $product,
            "qty"           => $qty,
            "attributes"    => $attributes,
            "bundleOptions" => $bundleOptions,
            "customOptions" => $customOptions,
            "item"          => $item
        ]);

        // Wishlist listens for this:
        Mage::dispatchEvent('checkout_cart_update_item_complete', [
            'product'  => $product,
            'request'  => $request,
            'response' => $response
        ]);
    }

    /**
     * Adds a new product to the cart.
     *
     * @param  Mage_Sales_Model_Quote
     * @param  Product  The product to update
     * @param  int      The new quantity
     * @param  mixed    Any product attributes (super_attributes)
     * @param  mixed    Any bundle selection options (bundle_option)
     * @param  mixed    Any custom options (options)
     */
    protected function addProductToCart($quote, $product, $qty, $attributes, $bundleOptions, $customOptions) {
        $request  = Mage::app()->getRequest();
        $response = Mage::app()->getResponse();

        Mage::dispatchEvent(self::EVENT_QUOTE_PRE_ADD_PRODUCT, [
            "quote"         => $quote,
            "product"       => $product,
            "qty"           => $qty,
            "attributes"    => $attributes,
            "bundleOptions" => $bundleOptions,
            "customOptions" => $customOptions,
        ]);

        try {
            $return = $quote->addProduct($product, new Varien_Object([
                "product"         => $product->getId(),
                "qty"             => $this->clampProductQty($quote, $product, $qty),
                "super_attribute" => $attributes,
                "bundle_option"   => $bundleOptions,
                "options"         => $customOptions,
            ]));

            if(is_string($return)) {
                throw new Exception($return);
            }
        } catch (Exception $e) {
            // Determnine if we return an error to the user or rethrow
            $this->handleProductException($e, $product);
        }

        // Not needed since no module is listening for it
        // Mage::dispatchEvent('checkout_cart_product_add_after', array('quote_item' => $result, 'product' => $product));

        // Same for this, only used in native controllers:
        // $session->setCartWasUpdated(true);
        // Used for crossell
        // $this->getCheckoutSession()->setLastAddedProductId($productId);

        Mage::dispatchEvent(self::EVENT_QUOTE_POST_ADD_PRODUCT, [
            "quote"         => $quote,
            "product"       => $product,
            "qty"           => $qty,
            "attributes"    => $attributes,
            "bundleOptions" => $bundleOptions,
            "customOptions" => $customOptions,
        ]);

        // Wishlist listens for this:
        Mage::dispatchEvent('checkout_cart_add_product_complete', [
            'product'  => $product,
            'request'  => $request,
            'response' => $response
        ]);
    }

    /**
     * Clamps the product qty to fall within the bounds of allowed quantities for the given product.
     *
     * @param  Mage_Sales_Model_Quote
     * @param  Mage_Catalog_Model_Product
     * @param  double
     * @return double
     */
    protected function clampProductQty($quote, $product, $qty) {
        if($product->isConfigurable() ||
           !$product->getStockItem() ||
           $quote->hasProductId($product->getId())) {
            return $qty;
        }

        return max($product->getStockItem()->getMinSaleQty(), $qty);
    }

    /**
     * Handles exceptions caused in $cart->addProduct or $cart->updateItem gracefully, produces a
     * Crossroads_API_ResponseException error message * with the correct HTTP-status and an errorCode if
     * the error is caused by user input. Will rethrow if it is a system error.
     *
     * @param  Exception
     * @param  Product  The product used when encountering the exception
     *
     * @throws Crossroads_API_ResponseException
     */
    protected function handleProductException($e, $product) {
        $msg = $e->getMessage();
        // Magento is not sane, it is using stringly TRANSLATED typed exceptions
        if($msg === Mage::helper('catalog')->__('Please specify the product\'s option(s).') ||
           $msg === Mage::helper('catalog')->__('Please specify the product\'s required option(s).'))  {
            throw Crossroads_API_ResponseException::create(400, "Please specify the product's option(s).", null, 2002);
        }

        if($msg === Mage::helper('checkout')->__('The product could not be found.')) {
            throw Crossroads_API_ResponseException::create(400, "Product coult not be found", null, 2009);
        }

        if($msg === Mage::helper('cataloginventory')->__('This product is currently out of stock.')) {
            throw Crossroads_API_ResponseException::create(400, "Product is currently out of stock", null, 2000);
        }

        if(stripos($msg, Mage::helper('sales')->__('Item qty declaration error.')) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('Item qty declaration error.\nThe requested quantity for \"%s\" is not available.', $product->getName())) !== false ||
           stripos($msg, Mage::helper('cataloginventory')->__('The requested quantity for "%s" is not available.', $product->getName())) !== false) {
            throw Crossroads_API_ResponseException::create(400, "Product quantity is not available.", null, 2001);
        }

        if(stripos($msg, strstr(Mage::helper('cataloginventory')->__('The maximum quantity allowed for purchase is %s.', "DEADBEEF"), "DEADBEEF", true)) !== false) {
            throw Crossroads_API_ResponseException::create(400, "Maximum quantity of requested product exceeded.", null, 2010);
        }

        if($msg === Mage::helper('bundle')->__('Required options are not selected.')) {
            throw Crossroads_API_ResponseException::create(400, "Required bundle options are not selected.", null, 2013);
        }

        if($msg === Mage::helper('cataloginventory')->__('This product with current option is not available')) {
            throw Crossroads_API_ResponseException::create(400, "Selected options are not available on the specified product", null, 2015);
        }

        throw $e;
    }
}