<?php

class Crossroads_API_DataController extends Crossroads_API_Controller_Resource
{
    public function getAll() {
        return [404];
    }

    public function getItem($id) {
        switch($id) {
        case "countries":
            return $this->getCountries();
        case "regions":
            return $this->getRegions();
        default:
            return [404];
        }
    }

    /**
     * @api {get} /data/countries List store-countries
     * @apiName dataListStoreCountries
     * @apiGroup Data
     *
     * @apiSuccess {String} code  Country code
     * @apiSuccess {String} name  Country name, translated
     * @apiSuccess {Boolean} requiresRegion If the country requires a region for addresses
     */
    public function getCountries() {
        $countries = Mage::getModel("directory/country")
            ->getResourceCollection()
            ->loadByStore()
            ->toOptionArray(false);
        $requiresRegion = array_filter(array_map("trim", explode(",", Mage::getStoreConfig(Mage_Directory_Helper_Data::XML_PATH_STATES_REQUIRED))));

        return [200, array_map(function($c) use($requiresRegion) {
            return [
                // ISO2 code
                "code"           => $c["value"],
                // Name in current store
                "name"           => $c["label"],
                "requiresRegion" => in_array($c["value"], $requiresRegion),
            ];
        }, $countries)];
    }

    /**
     * @api {get} /data/regions List regions for a specific country in the store
     * @apiName dataListStoreRegions
     * @apiGroup Data
     *
     * @apiParam {String}    countryCode
     * @apiSuccess {Integer} id
     * @apiSuccess {String}  code
     * @apiSuccess {String}  name
     */
    public function getRegions() {
        $countryCode = $this->getRequest()->getParam("countryCode");

        if( ! $countryCode) {
            throw Crossroads_API_ResponseException::create(400, "Missing countryCode parameter", null);
        }

        $country = Mage::getModel('directory/country')->loadByCode($countryCode);

        if( ! $country) {
            return [404];
        }

        return [200, array_values(array_map(function($r) {
            return [
                "id"   => (int)$r->getId(),
                "code" => $r->getCode(),
                "name" => $r->getName(),
            ];
        }, $country->getRegions()->getItems()))];
    }
}
