<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;

require_once("Mage.php");
require_once("Mage/Catalog/Model/Product.php");

require_once("Model/Serializer/Abstract.php");
require_once("Model/Serializer/Product/Abstract.php");
require_once("Model/Serializer/Product/Options.php");
require_once("Model/Serializer/Product/Stockinfo.php");
require_once("Model/Serializer/Groupprices.php");
require_once("Model/Serializer/Attributes.php");

/**
 * @covers Crossroads_API_Model_Serializer_Product_Options
 */
class Crossroads_API_Model_Serializer_Product_OptionsTest extends TestCase {
    public function setUp() {
        Mage::_clear();
    }

    public function testSimple() {
        $typeInst   = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getConfigurableAttributesAsArray", "getUsedProducts"])
                           ->getMock();
        $product    = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                           ->setMethods(["getTypeId"])
                           ->getMock();
        $groupprices = $this->createMock(Crossroads_API_Model_Serializer_Groupprices::class);
        $stockInfo   = $this->createMock(Crossroads_API_Model_Serializer_Product_Stockinfo::class);
        $attributes  = $this->createMock(Crossroads_API_Model_Serializer_Attributes::class);
        $smallImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $largeImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $origImage   = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $customOpts  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);

        $attributes->expects($this->never())->method("serializeItem");
        $customOpts->expects($this->never())->method("serializeItem");
        $groupprices->expects($this->never())->method("serializeItem");
        $largeImage->expects($this->never())->method("serializeItem");
        $origImage->expects($this->never())->method("serializeItem");
        $smallImage->expects($this->never())->method("serializeItem");
        $stockInfo->expects($this->never())->method("serializeItem");
        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue("simple"));

        $serializer = new Crossroads_API_Model_Serializer_Product_Options([
            "attributes"    => $attributes,
            "customOptions" => $customOpts,
            "groupPrices"   => $groupprices,
            "largeImage"    => $largeImage,
            "origImage"     => $origImage,
            "smallImage"    => $smallImage,
            "stockInfo"     => $stockInfo,
        ]);

        $this->assertSame(null, $serializer->serializeItem($product));
    }

    public function testEmpty() {
        $typeInst   = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getConfigurableAttributesAsArray", "getUsedProducts"])
                           ->getMock();
        $product    = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                           ->setMethods(["getTypeId", "getTypeInstance"])
                           ->getMock();
        $groupprices = $this->createMock(Crossroads_API_Model_Serializer_Groupprices::class);
        $stockInfo   = $this->createMock(Crossroads_API_Model_Serializer_Product_Stockinfo::class);
        $attributes  = $this->createMock(Crossroads_API_Model_Serializer_Attributes::class);
        $smallImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $largeImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $origImage   = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $customOpts  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);

        $attributes->expects($this->never())->method("serializeItem");
        $customOpts->expects($this->never())->method("serializeItem");
        $groupprices->expects($this->never())->method("serializeItem");
        $largeImage->expects($this->never())->method("serializeItem");
        $origImage->expects($this->never())->method("serializeItem");
        $smallImage->expects($this->never())->method("serializeItem");
        $stockInfo->expects($this->never())->method("serializeItem");
        $typeInst->expects($this->once())
                 ->method("getConfigurableAttributesAsArray")
                 ->with($this->identicalTo($product))
                 ->will($this->returnValue([]));
        $typeInst->expects($this->once())
                 ->method("getUsedProducts")
                 ->with($this->equalTo(null), $this->identicalTo($product))
                 ->will($this->returnValue([]));
        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue("configurable"));
        $product->expects($this->any())
                ->method("getTypeInstance")
                ->with($this->equalTo(true))
                ->will($this->returnValue($typeInst));

        $serializer = new Crossroads_API_Model_Serializer_Product_Options([
            "attributes"    => $attributes,
            "customOptions" => $customOpts,
            "groupPrices"   => $groupprices,
            "largeImage"    => $largeImage,
            "origImage"     => $origImage,
            "smallImage"    => $smallImage,
            "stockInfo"     => $stockInfo,
        ]);

        $this->assertSame([], $serializer->serializeItem($product));
    }

    /**
     * @dataProvider serializeItemMismatchOptions
     */
    public function testSerializeItemMismatch($options, $usedProducts, $expected) {
        $attrObj    = new stdClass;
        $backObj    = new stdClass;
        $groupObj   = new stdClass;
        $manageObj  = new stdClass;
        $qtyObj     = new stdClass;
        $typeInst   = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getConfigurableAttributesAsArray", "getUsedProducts"])
                           ->getMock();
        $product    = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                           ->setMethods(["getTypeId", "getMsrp", "getTypeInstance"])
                           ->getMock();
        $groupprices = $this->createMock(Crossroads_API_Model_Serializer_Groupprices::class);
        $stockInfo   = $this->createMock(Crossroads_API_Model_Serializer_Product_Stockinfo::class);
        $attributes  = $this->createMock(Crossroads_API_Model_Serializer_Attributes::class);
        $smallImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $largeImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $origImage   = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $customOpts  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);

        $attributes->expects($this->never())->method("serializeItem");
        $customOpts->expects($this->never())->method("serializeItem");
        $groupprices->expects($this->never())->method("serializeItem");
        $largeImage->expects($this->never())->method("serializeItem");
        $origImage->expects($this->never())->method("serializeItem");
        $smallImage->expects($this->never())->method("serializeItem");
        $stockInfo->expects($this->never())->method("serializeItem");

        $typeInst->expects($this->once())
                 ->method("getConfigurableAttributesAsArray")
                 ->with($this->identicalTo($product))
                 ->will($this->returnValue($options));
        $typeInst->expects($this->once())
                 ->method("getUsedProducts")
                 ->with($this->equalTo(null), $this->identicalTo($product))
                 ->will($this->returnValue($usedProducts));
        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue("configurable"));
        $product->expects($this->any())
                ->method("getTypeInstance")
                ->with($this->equalTo(true))
                ->will($this->returnValue($typeInst));
        $product->expects($this->never())
                ->method("getMsrp");

        $serializer = new Crossroads_API_Model_Serializer_Product_Options([
            "attributes"    => $attributes,
            "customOptions" => $customOpts,
            "groupPrices"   => $groupprices,
            "largeImage"    => $largeImage,
            "origImage"     => $origImage,
            "smallImage"    => $smallImage,
            "stockInfo"     => $stockInfo,
        ]);

        $this->assertSame($expected, $serializer->serializeItem($product));
    }

    public function serializeItemMismatchOptions() {
        $pMock = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData"])
                      ->getMock();
        $pMock->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(false));
        $pMock->expects($this->any())
              ->method("getData")
              ->with($this->equalTo("test_attr"))
              ->will($this->returnValue("12345"));

        $pMock2 = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData"])
                      ->getMock();
        $pMock2->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(true));
        $pMock2->expects($this->any())
              ->method("getData")
              ->with($this->equalTo("test_attr"))
              ->will($this->returnValue("12345"));

        return [
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => []
                ]
            ],
            [],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => [[
                        "value_index" => 23,
                    ]]
                ]
            ],
            [],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => [[
                        "value_index" => 23,
                    ]]
                ]
            ],
            [$pMock],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => [[
                        "value_index" => 23,
                    ]]
                ]
            ],
            [$pMock, $pMock2],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => [[
                        "value_index" => 23,
                    ]]
                ]
            ],
            [$pMock2],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
            [[
                [
                    "attribute_id"   => "23",
                    "attribute_code" => "test_attr",
                    "store_label"    => null,
                    "frontend_label" => null,
                    "label"          => "Test",
                    "use_default"    => "32",
                    "position"       => "5",
                    "values"         => [[
                        "value_index" => 23,
                    ]]
                ]
            ],
            [$pMock, $pMock2],
            [
                [
                    "id"           => 23,
                    "code"         => "test_attr",
                    "title"        => "Test",
                    "useAsDefault" => true,
                    "position"     => 5,
                    "values"       => [],
                ]
            ]],
        ];
    }

    public function testSerializeItem() {
        $attr1      = new stdClass;
        $attr2      = new stdClass;
        $custom1    = new stdClass;
        $groupPricesObj = new stdClass;
        $custom2    = new stdClass;
        $large1     = new stdClass;
        $large2     = new stdClass;
        $orig1      = new stdClass;
        $orig2      = new stdClass;
        $small1     = new stdClass;
        $small2     = new stdClass;
        $stock1     = new stdClass;
        $stock2     = new stdClass;
        $back1      = new stdClass;
        $back2      = new stdClass;
        $group1     = new stdClass;
        $group2     = new stdClass;
        $manage1    = new stdClass;
        $manage2    = new stdClass;
        $qty1       = new stdClass;
        $qty2       = new stdClass;
        $typeInst   = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getConfigurableAttributesAsArray", "getUsedProducts"])
                           ->getMock();
        $product    = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                           ->setMethods(["getTypeId", "getMsrp", "getTypeInstance"])
                           ->getMock();
        $groupprices = $this->createMock(Crossroads_API_Model_Serializer_Groupprices::class);
        $stockInfo   = $this->createMock(Crossroads_API_Model_Serializer_Product_Stockinfo::class);
        $attributes  = $this->createMock(Crossroads_API_Model_Serializer_Attributes::class);
        $smallImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $largeImage  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $origImage   = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $customOpts  = $this->createMock(Crossroads_API_Model_Serializer_Abstract::class);
        $pMock = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData"])
                      ->getMock();
        $pMock->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(false));
        $pMock->expects($this->any())
              ->method("getData")
              ->with($this->equalTo("test_attr"))
              ->will($this->returnValue("12345"));
        $p1 = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData"])
                      ->getMock();
        $p2 = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData", "getSku", "isInStock", "getIsSalable", "getSpecialPrice", "getFinalPrice", "getPrice", "getMsrp"])
                      ->getMock();
        $p3 = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                      ->setMethods(["isSaleable", "getData", "getSku", "isInStock", "getIsSalable", "getSpecialPrice", "getFinalPrice", "getPrice", "getMsrp"])
                      ->getMock();
        $p1->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(false));
        $p1->expects($this->any())
              ->method("getData")
              ->with($this->equalTo("test_attr"))
              ->will($this->returnValue("23"));
        $p2->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(true));
        $p2->expects($this->any())
              ->method("getData")
              ->with($this->logicalOr($this->equalTo("test_attr"), $this->equalTo("group_price")))
              ->will($this->returnValueMap([
                  ["test_attr", "23"],
                  ["group_price", null]
              ]));
        $p2->expects($this->once())->method("getSku")->will($this->returnValue("SKUA"));
        $p2->expects($this->once())->method("isInStock")->will($this->returnValue(true));
        $p2->expects($this->once())->method("getIsSalable")->will($this->returnValue(true));
        $p2->expects($this->any())->method("getSpecialPrice")->will($this->returnValue("1234"));
        $p2->expects($this->once())->method("getFinalPrice")->will($this->returnValue("1234.5"));
        $p2->expects($this->any())->method("getMsrp")->will($this->returnValue("432"));
        $p3->expects($this->any())
              ->method("isSaleable")
              ->will($this->returnValue(true));
        $p3->expects($this->any())
              ->method("getData")
              ->with($this->logicalOr($this->equalTo("test_attr"), $this->equalTo("group_price")))
              ->will($this->returnValueMap([
                  ["test_attr",   "45"],
                  ["group_price", $groupPricesObj]
              ]));
        $p3->expects($this->once())->method("getSku")->will($this->returnValue("SKUB"));
        $p3->expects($this->once())->method("isInStock")->will($this->returnValue(false));
        $p3->expects($this->once())->method("getIsSalable")->will($this->returnValue(true));
        $p3->expects($this->any())->method("getSpecialPrice")->will($this->returnValue(null));
        $p3->expects($this->once())->method("getFinalPrice")->will($this->returnValue("14.5"));
        $p3->expects($this->any())->method("getMsrp")->will($this->returnValue(null));

        $attributes->expects($this->exactly(2))
                   ->method("serializeItem")
                   ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                   ->will($this->returnValueMap([[$p2, $attr1], [$p3, $attr2]]));
        $customOpts->expects($this->exactly(2))
                   ->method("serializeItem")
                   ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                   ->will($this->returnValueMap([[$p2, $custom1], [$p3, $custom2]]));
        $groupprices->expects($this->exactly(2))
                    ->method("serializeItem")
                    ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                    ->will($this->returnValueMap([[$p2, $group1], [$p3, $group2]]));
        $largeImage->expects($this->exactly(2))
                   ->method("serializeItem")
                   ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                   ->will($this->returnValueMap([[$p2, $large1], [$p3, $large2]]));
        $origImage->expects($this->exactly(2))
                  ->method("serializeItem")
                  ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                  ->will($this->returnValueMap([[$p2, $orig1], [$p3, $orig2]]));
        $smallImage->expects($this->exactly(2))
                   ->method("serializeItem")
                   ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                   ->will($this->returnValueMap([[$p2, $small1], [$p3, $small2]]));
        $stockInfo->expects($this->exactly(2))
                  ->method("serializeItem")
                  ->with($this->logicalOr($this->identicalTo($p2), $this->identicalTo($p3)))
                  ->will($this->returnValueMap([[$p2, [
                      "stockQty"        => $qty1,
                      "stockBackorders" => $back1,
                      "stockManage"     => $manage1,
                  ]], [$p3, [
                      "stockQty"        => $qty2,
                      "stockBackorders" => $back2,
                      "stockManage"     => $manage2,
                  ]]]));

        $typeInst->expects($this->once())
                 ->method("getConfigurableAttributesAsArray")
                 ->with($this->identicalTo($product))
                 ->will($this->returnValue([
                    [
                        "attribute_id"   => "23",
                        "attribute_code" => "test_attr",
                        "store_label"    => null,
                        "frontend_label" => null,
                        "label"          => "Test",
                        "use_default"    => "32",
                        "position"       => "5",
                        "values"         => [
                            [
                                "value_index"   => "23",
                                "store_label"   => "a",
                                "label"         => "b",
                                "default_label" => "c",
                                "is_percent"    => "0",
                            ],
                            [
                                "value_index"   => "45",
                                "store_label"   => null,
                                "label"         => "B",
                                "default_label" => "C",
                                "is_percent"    => "1",
                            ],
                        ]
                    ],
                    [
                        "attribute_id"   => "42",
                        "attribute_code" => "test",
                        "store_label"    => null,
                        "frontend_label" => "Another",
                        "label"          => "A",
                        "use_default"    => null,
                        "position"       => null,
                        "values"         => [

                        ]
                    ],
                 ]));
        $typeInst->expects($this->once())
                 ->method("getUsedProducts")
                 ->with($this->equalTo(null), $this->identicalTo($product))
                 ->will($this->returnValue([
                     $p1,
                     $p2,
                     $p3,
                 ]));
        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue("configurable"));
        $product->expects($this->any())
                ->method("getTypeInstance")
                ->with($this->equalTo(true))
                ->will($this->returnValue($typeInst));
        $product->expects($this->never())
                ->method("getMsrp");

        $serializer = new Crossroads_API_Model_Serializer_Product_Options([
            "attributes"    => $attributes,
            "customOptions" => $customOpts,
            "groupPrices"   => $groupprices,
            "largeImage"    => $largeImage,
            "origImage"     => $origImage,
            "smallImage"    => $smallImage,
            "stockInfo"     => $stockInfo,
        ]);

        $this->assertSame([
            [
                "id"           => 23,
                "code"         => "test_attr",
                "title"        => "Test",
                "useAsDefault" => true,
                "position"     => 5,
                "values" => [
                    [
                        "id"              => 23,
                        "sku"             => "SKUA",
                        "smallImage"      => $small1,
                        "largeImage"      => $large1,
                        "originalImage"   => $orig1,
                        "label"           => "a",
                        "isPercent"       => false,
                        "isInStock"       => true,
                        "isSalable"       => true,
                        "groupPrices"     => $group1,
                        "specialPrice"    => 1234.0,
                        "price"           => 1234.5,
                        "originalPrice"   => 0.0,
                        "msrp"            => 432.0,
                        "customOptions"   => $custom1,
                        "attributes"      => $attr1,
                        "stockQty"        => $qty1,
                        "stockBackorders" => $back1,
                        "stockManage"     => $manage1,
                    ],
                    [
                        "id"               => 45,
                         "sku"             => "SKUB",
                         "smallImage"      => $small2,
                         "largeImage"      => $large2,
                         "originalImage"   => $orig2,
                         "label"           => "B",
                         "isPercent"       => true,
                         "isInStock"       => false,
                         "isSalable"       => true,
                         "groupPrices"     => $group2,
                         "specialPrice"    => null,
                         "price"           => 14.5,
                         "originalPrice"   => 0.0,
                         "msrp"            => null,
                         "customOptions"   => $custom2,
                         "attributes"      => $attr2,
                         "stockQty"        => $qty2,
                         "stockBackorders" => $back2,
                         "stockManage"     => $manage2,
                     ],
                ],
            ],
            [
                "id"           => 42,
                "code"         => "test",
                "title"        => "Another",
                "useAsDefault" => false,
                "position"     => 0,
                "values"       => [],
            ],
        ], $serializer->serializeItem($product));
    }
}
