<?php

class Crossroads_API_CategoriesController extends Crossroads_API_Controller_Resource
{
    /**
     * @api {get} /categories Get all categories
     * @apiName getCategories
     * @apiGroup Categories
     *
     * @apiParam {Boolean}        includeImages=false If set to "true" it will include images.
     * @apiParam {Boolean}        [includeInMenu] If set to "true" it will only list categories visible in menu
     * @apiParam {Integer/String} [parentId]      Filters the list of categories by parent category, the string "root" is a
     *                                            special parent id which refers to the store root category.
     *
     * @apiSuccess {Object[]} categories Array of the categories
     * @apiSuccess {String} categories.id Category ID
     * @apiSuccess {String} categories.name Name of the category
     * @apiSuccess {String} categories.description Category description
     * @apiSuccess {String} categories.metaDescription
     * @apiSuccess {String} categories.urlKey
     * @apiSuccess {String} categories.url
     * @apiSuccess {Integer} categories.position
     * @apiSuccess {Integer} categories.level
     * @apiSuccess {Boolean} categories.isAnchor
     * @apiSuccess {String} categories.path
     * @apiSuccess {Integer} categories.parentId
     * @apiSuccess {Boolean} categories.includeInMenu
     * @apiSuccess {Integer} [categories.productCount] Will be present if the setting is turned on in config
     */
    public function getAll()
    {
        $req           = $this->getRequest();
        $store         = Mage::app()->getStore();
        $factory       = Mage::getModel("API/factory");
        $rootId        = $store->getRootCategoryId();
        $includeImages = $req->has("includeImages") ? strtolower($req->get("includeImages")) === "true" : false;
        $includeInMenu = $req->has("includeInMenu") ? strtolower($req->get("includeInMenu")) === "true" : false;
        $parentId      = false;

        if($req->has("parentId")) {
            if(ctype_digit($req->get("parentId"))) {
                $parentId = $req->get("parentId");
            }
            elseif(strtolower($req->get("parentId")) === "root") {
                $parentId = $rootId;
            }
            else {
                throw Crossroads_API_ResponseException::create(400, "Parameter parentId must either be an integer or the string 'root'.", 1002);
            }
        }

        $catSer     = $factory->createCategorySerializer($store, $includeImages);
        // TODO: Make a collection factory for this
        $categories = Mage::getModel('catalog/category')
            ->getCollection()
            ->setStoreId($store->getId())
            ->addFieldToFilter('is_active', 1)
            ->addAttributeToFilter('path', ['like' => "1/{$rootId}/%"])
            ->addAttributeToSelect('*')
            ->addUrlRewriteToResult()
            ->addAttributeToSort("position");

        if($parentId) {
            $categories->addFieldToFilter("parent_id", $parentId);
        }

        if($includeInMenu) {
            $categories->addFieldToFilter("include_in_menu", $includeInMenu);
        }

        return [200, $catSer->mapCollection($categories)];
    }

    /**
     * @api {get} /categories/:id Get category
     * @apiName getCategory
     * @apiGroup Categories
     *
     * @apiParam {Integer} id Category ID
     * @apiParam {Integer} [limit=20] The pagination size, max 100
     * @apiParam {Integer} [page=1] The pagination page
     *
     * @apiSuccess {Object} category Category
     * @apiSuccess {String} category.id Category ID
     * @apiSuccess {String} category.name Name of the category
     * @apiSuccess {String} category.description Category description
     * @apiSuccess {String} category.metaDescription
     * @apiSuccess {String} category.urlKey
     * @apiSuccess {String} category.url
     * @apiSuccess {String} category.metaTitle
     * @apiSuccess {Integer} category.position
     * @apiSuccess {Array} category.images
     * @apiSuccess {Object} category.images.small
     * @apiSuccess {String} category.images.small.url
     * @apiSuccess {String} category.images.small.dataUrl
     * @apiSuccess {Object} category.images.medium
     * @apiSuccess {String} category.images.medium.url
     * @apiSuccess {String} category.images.medium.dataUrl
     * @apiSuccess {Object} category.images.large
     * @apiSuccess {String} category.images.large.url
     * @apiSuccess {String} category.images.large.dataUrl
     * @apiSuccess {Integer} category.level
     * @apiSuccess {Boolean} category.isAnchor
     * @apiSuccess {String} category.path
     * @apiSuccess {Integer} category.parentId
     * @apiSuccess {Boolean} category.includeInMenu
     * @apiSuccess {Integer} [category.productCount] Will be present if the setting is turned on in config
     *
     * @apiSuccess {Object} category.hero
     * @apiSuccess {String} category.hero.heading
     * @apiSuccess {String} category.hero.description
     * @apiSuccess {String} category.hero.offerIcon
     * @apiSuccess {Object} category.hero.button
     * @apiSuccess {String} category.hero.button.text
     * @apiSuccess {String} category.hero.button.link
     *
     * @apiSuccess {Object[]} products     Array of the products in the category
     * @apiSuccess {Integer}  products.id Product id
     * @apiSuccess {String}   products.type Product type
     * @apiSuccess {String}   products.sku
     * @apiSuccess {String}   products.urlKey
     * @apiSuccess {Double}   products.price
     * @apiSuccess {Double}   [products.specialPrice]
     * @apiSuccess {Object[]} [products.groupPrices]          Group prices, if enabled
     * @apiSuccess {String}   [products.groupPrices.groupCode]
     * @apiSuccess {Double}   [products.groupPrices.price]
     * @apiSuccess {Double}   [stockQty]               Amount in stock, null if information not available
     * @apiSuccess {Double}   products.msrp
     * @apiSuccess {Double}   products.discountPercent
     * @apiSuccess {String}   products.shortDescription
     * @apiSuccess {Object}   products.attributes              Key value pairs of the selected attributes in API_section/product_list/attributes
     * @apiSuccess {String}   products.smallImage
     * @apiSuccess {Boolean}  products.isSalable
     * @apiSuccess {Integer}  products.position   Null if product list is not a category
     * @apiSuccess {Object[]} [products.options]  Null if simple product, object if complex, can be turned off in list view through System Configuration
     */
    public function getItem($id)
    {
        // TODO: Verify that the category is available in the store
        $store    = Mage::app()->getStore();
        $req      = $this->getRequest();
        $category = Mage::getModel('catalog/category')->load($id);

        if( ! $category || !$category->getId() || !$category->getIsActive()) {
            return [404];
        }

        $factory    = Mage::getModel("API/factory");
        $prodSer    = $factory->createProductListSerializer($store);
        $catSer     = $factory->createCategorySerializer($store, true);
        $collection = Mage::getModel("API/collection_product_category")
            ->setStore($store)
            ->setCategory($category)
            ->setPage($req->getQuery("page"))
            ->setLimit($req->getQuery("limit"))
            ->createCollection();

        return [200, [
            "category" => $catSer->serializeItem($category),
            "products" => $prodSer->mapCollection($collection),
        ]];
    }
}
