<?php

class Crossroads_API_Model_Serializer_Product_Detail extends Crossroads_API_Model_Serializer_Product_Abstract {
    /**
     * Executed after the basic product associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_POST_DATA_PREPARE      = "crossroads_api_product_post_data_prepare";

    protected $attributes    = null;
    protected $bundle        = null;
    protected $customOptions = null;
    protected $groupPrices   = null;
    protected $largeImage    = null;
    protected $mediaGallery  = null;
    protected $options       = null;
    protected $origImage     = null;
    protected $related       = null;
    protected $smallImage    = null;
    protected $stockInfo     = null;

    public function __construct(array $params = []) {
        $this->attributes    = $params["attributes"];
        $this->bundle        = $params["bundle"];
        $this->customOptions = $params["customOptions"];
        $this->groupPrices   = $params["groupPrices"];
        $this->largeImage    = $params["largeImage"];
        $this->mediaGallery  = $params["mediaGallery"];
        $this->options       = $params["options"];
        $this->origImage     = $params["origImage"];
        $this->related       = $params["related"];
        $this->smallImage    = $params["smallImage"];
        $this->stockInfo     = $params["stockInfo"];

        parent::__construct($params);
    }

    public function serializeItem($product) {
        list($price, $originalPrice) = $product->getTypeId() === self::PRODUCT_BUNDLE ?
            $product->getPriceModel()->getTotalPrices($product, null, true) :
            [$product->getFinalPrice(), $product->getPrice()];
        $msrp            = $this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$price / $msrp)) * 100)
            : null;

        $inStock = $product->getIsInStock() > 0;

        // when the configurable products children is out of stock the price
        // gets calculated to "0". To avoid loading each child we simply assume
        // the product is out of stock.
        if ($product->getTypeId() === self::PRODUCT_CONFIGURABLE && !$price) {
            $inStock = false;
        }

        $productData = new Varien_Object(array_merge([
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),

            "price"            => (double)$price,
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            // Included normally when a product is ->load()ed
            "groupPrices"      => $this->groupPrices->serializeItem($product),
            "categoryIds"      => array_map(function($id) { return (int)$id; }, $product->getCategoryIds() ?: []),
            "originalPrice"    => (double)$originalPrice,
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "metaDescription"  => $product->getMetaDescription(),
            "shortDescription" => $product->getShortDescription(),
            "description"      => $product->getDescription(),
            "urlKey"           => $product->getUrlKey(),
            "smallImage"       => $this->smallImage->serializeItem($product),
            "largeImage"       => $this->largeImage->serializeItem($product),
            "originalImage"    => $this->origImage->serializeItem($product),
            "isInStock"        => $inStock,
            "isSalable"        => $product->getIsSalable() > 0,
            "options"          => $this->options->serializeItem($product),
            "selections"       => $this->bundle->serializeItem($product),
            "relatedProducts"  => $this->related->serializeItem($product),
            "mediaGallery"     => $this->mediaGallery->serializeItem($product),
            "customOptions"    => $this->customOptions->serializeItem($product),
            "attributes"       => $this->attributes->serializeItem($product),
        ], $this->stockInfo->serializeItem($product)));

        Mage::dispatchEvent(self::EVENT_POST_DATA_PREPARE, [
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }
}
