<?php

class Crossroads_API_StaticblocksController extends Crossroads_API_Controller_Resource {
    public function getAll() {
        return [404];
    }

    /**
     * @api {get} /staticblocks/:id  Get static block
     * @apiName getStaticBlock
     * @apiGroup Static Blocks
     * @apiDescription If the `:id` parameter contains a comma (`,`), then the `:id` is split
     *                 and the blocks matching the mutliple ids are fetched in a list.
     *
     * @apiParam {String} id  Block identifier
     *
     * @apiSuccess {String}  id        Block identifier
     * @apiSuccess {String}  title
     * @apiSuccess {Integer} blockId
     * @apiSuccess {String}  content   HTML content
     * @apiSuccess {String}  createdAt ISO 8601 timestamp, UTC
     * @apiSuccess {String}  updatedAt ISO 8601 timestamp, UTC
     *
     * @apiSuccessExample {json} Example
     * {"id":"footer_links","title":"Footer Links","blockId":1,"content":"\n<ul>\n    <li><a href=\"http:\/\/example.com\/about-magento-demo-store\">About Us<\/a><\/li>\n    <li><a href=\"http:\/\/example.com\/customer-service\">Customer Service<\/a><\/li>\n<li class=\"last privacy\"><a href=\"http:\/\/example.com\/privacy-policy-cookie-restriction-mode\">Privacy Policy<\/a><\/li>\r\n<\/ul>","createdAt":"2017-03-16T09:02:00Z","updatedAt":"2017-03-16T09:02:01Z"}
     * @apiSuccessExample {json} Multiple items
     * [{"id":"default_banner","title":"OutOfHome Default Banner SE","blockId":93,"content":"<div class=\"block-content height-2-block\"><a title=\"Caliente\" href=\"\/kalla-drycker\/kaffedryck\" target=\"_self\"><img title=\"L&ouml;fbergs ICE Slim\" src=\"http:\/\/outofhome3-sek.tld\/media\/wysiwyg\/OutOfHome\/banner\/banner_lofbergs-takeover.jpg\" alt=\"L&ouml;fbergs ICE SLIM\" width=\"220\" height=\"689\" \/><\/a><\/div>","createdAt":"2014-03-28T09:09:35Z","updatedAt":"2016-09-29T15:23:20Z"},{"id":"coffe_banner","title":"OutOfHome Coffe banner SE","blockId":98,"content":"<div class=\"block-content height-2-block\"><a title=\"Kaffe | outofhome.nu\" href=\"\/kaffe\" target=\"_self\"><img src=\"http:\/\/outofhome3-sek.tld\/media\/wysiwyg\/OutOfHome\/banner\/ooh_banner_kaffe_2_del.png\" alt=\"Kaffe | outofhome.nu\" \/><\/a><\/div>","createdAt":"2014-05-15T11:45:49Z","updatedAt":"2016-05-31T09:39:39Z"}]
     */
    public function getItem($id) {
        $store = Mage::app()->getStore();

        if(strpos($id, ",") !== false) {
            $blocks = Mage::getModel("cms/block")
                ->getCollection()
                ->addStoreFilter($store)
                ->addFieldToFilter("identifier", [ "in" => array_filter(array_map("trim", explode(",", $id))) ])
                ->getItems();

            return [200, array_values(array_map([$this, "prepareBlock"], $blocks))];
        }

        $block = Mage::getModel("cms/block")
            ->setStoreId($store->getId())
            ->load($id);

        if( ! $block || ! $block->getBlockId() || ! $block->getIsActive()) {
            return [404];
        }

        return [200, $this->prepareBlock($block)];
    }

    protected function prepareBlock($block) {
        $processor = Mage::helper("cms")->getPageTemplateProcessor();
        $html      = $processor->filter($block->getContent());

        return [
            "id"        => $block->getIdentifier(),
            "title"     => $block->getTitle(),
            // We do not use this as the id since we can have multiple blocks mapping to the same identifier
            // More useful for debugging
            "blockId"   => (int)$block->getBlockId(),
            "content"   => $html,
            "createdAt" => gmdate("Y-m-d\TH:i:s\Z", strtotime($block->getCreationTime())),
            "updatedAt" => gmdate("Y-m-d\TH:i:s\Z", strtotime($block->getUpdateTime())),
        ];
    }
}
