<?php

/**
 * Converts a Magento product with loaded group prices to a list with human-readable
 * customer groups filtered to only contain relevant prices for the given website.
 *
 * NOTE: If the product does not have the `group_price` property loaded it will
 *       result in an empty array.
 */
class Crossroads_API_Model_Serializer_Groupprices extends Crossroads_API_Model_Serializer_Abstract {
    /**
     * Customer group id => customer group name map, null if not yet loaded.
     *
     * @var array
     */
    protected $customerGroups = [];
    protected $websiteId      = 0;

    public function __construct(array $params = []) {
        array_key_exists("websiteId", $params) && $this->websiteId = $params["websiteId"];

        $this->customerGroups = array_map(function($group) {
            return $group->getCustomerGroupCode();
        }, Mage::getModel("customer/group")->getCollection()->getItems());

        parent::__construct($params);
    }

    public function getCustomerGroups() {
        return $this->customerGroups;
    }

    /**
     * @param  Mage_Catalog_Model_Product
     * @return Array<["groupCode" => string|null, "price" => double]>
     */
    public function serializeItem($product) {
        return $this->serializeGroupPrices($product->getData("group_price") ?: []);
    }

    /**
     * @param  Array<["website_id" => int|null, "cust_group" => int, "price => double]>
     * @return Array<["groupCode" => string|null, "price" => double]>
     */
    protected function serializeGroupPrices($groupPrices) {
        $customerGroups = $this->getCustomerGroups();

        return array_values(array_filter(array_map(function($price) use($customerGroups) {
            if($price["website_id"] != $this->websiteId && $price["website_id"] != 0) {
                return null;
            }

            return [
                "groupCode" => array_key_exists($price["cust_group"], $customerGroups) ? $customerGroups[$price["cust_group"]] : null,
                "price"     => (double)$price["price"]
            ];
        }, $groupPrices)));
    }
}
