<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;

require_once("Mage.php");
require_once("Mage/Catalog/Model/Product.php");

require_once("Model/Serializer/Abstract.php");
require_once("Model/Serializer/Product/Abstract.php");

/**
 * @covers Crossroads_API_Model_Serializer_Product_Abstract
 */
class Crossroads_API_Model_Serializer_Product_AbstractTest extends TestCase {
    /**
     * Utility function to test protected methods later used in child-classes.
     */
    protected static function callProtected(Crossroads_API_Model_Serializer_Product_Abstract $obj, $name, $args = []) {
        $method = (new ReflectionClass(Crossroads_API_Model_Serializer_Product_Abstract::class))
            ->getMethod($name);

        $method->setAccessible(true);

        return $method->invokeArgs($obj, $args);
    }

    /**
     * @dataProvider msrpSimpleAttributes
     */
    public function testGetMsrpSimple($typeId, $msrp, $expected) {
        $product = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                        ->setMethods(["getTypeId", "getMsrp"])
                        ->getMock();

        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue($typeId));
        $product->expects($this->once())
                ->method("getMsrp")
                ->will($this->returnValue($msrp));

        $serializer = $this->getMockBuilder(Crossroads_API_Model_Serializer_Product_Abstract::class)
                           ->setMethods(["serializeItem"])
                           ->getMock();
        $serializer->expects($this->never())
                   ->method("serializeItem");

        $this->assertSame($expected, self::callProtected($serializer, "getMsrp", [$product]));
    }

    public function msrpSimpleAttributes() {
        return [
            ["simple", 12, 12.0],
            ["simple", 24.6, 24.6],
            ["simple", "0", 0.0],
            ["simple", "12", 12.0],
            ["simple", "", 0.0],
            ["simple", "foo", 0.0],
            ["bundle", 24.6, 24.6],
            ["virtual", 24.6, 24.6],
        ];
    }

    /**
     * @dataProvider getMsrpConfigurableAttributes
     */
    public function testGetMsrpConfigurable($options, $usedProducts, $expected) {
        $typeInst   = $this->getMockBuilder(stdClass::class)
                           ->setMethods(["getConfigurableAttributesAsArray", "getUsedProducts"])
                           ->getMock();
        $product    = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                           ->setMethods(["getTypeId", "getMsrp", "getTypeInstance"])
                           ->getMock();
        $serializer = $this->getMockBuilder(Crossroads_API_Model_Serializer_Product_Abstract::class)
                           ->setMethods(["serializeItem"])
                           ->getMock();

        $typeInst->expects($this->once())
                 ->method("getConfigurableAttributesAsArray")
                 ->with($this->identicalTo($product))
                 ->will($this->returnValue($options));
        $typeInst->expects($this->once())
                 ->method("getUsedProducts")
                 ->with($this->equalTo(null), $this->identicalTo($product))
                 ->will($this->returnValue($usedProducts));
        $product->expects($this->once())
                ->method("getTypeId")
                ->will($this->returnValue("configurable"));
        $product->expects($this->any())
                ->method("getTypeInstance")
                ->with($this->equalTo(true))
                ->will($this->returnValue($typeInst));
        $product->expects($this->never())
                ->method("getMsrp");

        $serializer->expects($this->never())
                   ->method("serializeItem");

        $this->assertSame($expected, self::callProtected($serializer, "getMsrp", [$product]));
    }

    public function getMsrpConfigurableAttributes() {
        $mkUsedProd = function($saleable, $key, $value, $msrp) {
            $prod = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                         ->setMethods(["getMsrp", "getData", "isSaleable"])
                         ->getMock();

            $prod->expects($this->any())
                 ->method("getData")
                 ->with($this->logicalOr($this->equalTo($key), $this->equalTo("n")))
                 ->will($this->returnValueMap([
                     [$key, $value],
                     ["n",  false],
                 ]));
            $prod->expects($this->any())
                 ->method("isSaleable")
                 ->will($this->returnValue($saleable));
            $prod->expects($this->any())
                 ->method("getMsrp")
                 ->will($this->returnValue($msrp));

            return $prod;
        };

        return [
            [[], [], null],
            [[], [$mkUsedProd(false, "a", "b", 23)], null],
            [[], [$mkUsedProd(true, "a", "b", 23)], null],
            [[["values" => []]], [$mkUsedProd(true, "a", "b", 23)], null],
            [
                [["attribute_code" => "a", "values" => [["value_index" => "b"]]]],
                [$mkUsedProd(true, "a", "c", 23)],
                null,
            ],
            [
                [["attribute_code" => "a", "values" => [["value_index" => "b"], ["value_index" => "c"]]]],
                [$mkUsedProd(false, "a", "c", 23)],
                null,
            ],
            [
                [["attribute_code" => "a", "values" => [["value_index" => "b"]]]],
                [$mkUsedProd(true, "a", "b", 23)],
                23.0,
            ],
            [
                [["attribute_code" => "a", "values" => [["value_index" => "b"], ["value_index" => "c"]]]],
                [$mkUsedProd(true, "a", "b", 13.5), $mkUsedProd(true, "a", "c", 23)],
                13.5,
            ],
            [
                [["attribute_code" => "a", "values" => [["value_index" => "b"], ["value_index" => "c"]]]],
                [$mkUsedProd(false, "a", "b", 13), $mkUsedProd(true, "a", "c", 23)],
                23.0,
            ],
            [
                [["attribute_code" => "n", "values" => [["value_index" => "b"], ["value_index" => "c"]]]],
                [$mkUsedProd(true, "a", "b", 13), $mkUsedProd(true, "a", "c", 23)],
                null,
            ],
        ];
    }
}
