<?php

/**
 * Validation helper, also providing normalization features.
 *
 * Supported types:
 *
 *  * Associative arrays: `validateMap` with `[key => valueSpecification, ...]`
 *  * Strings: `validate` with `["type" => "string"]`, optional `required: true`
 */
class Crossroads_API_Helper_Validation {
    public static function isRequired($specification) {
        return array_key_exists("required", $specification) && $specification["required"];
    }

    /**
     * Validates a map (associative array).
     *
     * @param  Specification of the array
     * @param  Candidate to validate
     * @param  If set to true keys not present in the specification will be considered invalid.
     * @return Crossroads_API_Helper_Validation_Result
     */
    public function validateMap($specification, $candidate, $strict = false) {
        if(!is_array($specification)) {
            throw new Exception("Specification for map must be an associative array.");
        }

        if(!is_array($candidate)) {
            return new Crossroads_API_Helper_Validation_Result(["Expected array got '".gettype($candidate)."'."]);
        }

        if($strict) {
            $diff = array_diff(array_keys($candidate), array_keys($specification));

            if(!empty($diff)) {
                return new Crossroads_API_Helper_Validation_Result(array_map(function($k) {
                    return "Unknown key '$k'.";
                }, $diff));
            }
        }

        $msgs       = [];
        $normalized = [];

        foreach($specification as $k => $v) {
            if(!is_array($v)) {
                throw new Exception("Invalid specification for key '$k', not associative array.");
            }

            if(!array_key_exists($k, $candidate)) {
                if(self::isRequired($v)) {
                    $msgs[] = "Missing key '$k'.";
                }

                continue;
            }

            $r = $this->validateItem($v, $candidate[$k]);

            $msgs = array_merge($msgs, array_map(function($m) use($k) {
                return $k.": ".$m;
            }, $r->getMessages()));

            $normalized[$k] = $r->getNormalizedData();
        }

        return new Crossroads_API_Helper_Validation_Result($msgs, $normalized);
    }

    /**
     * Validates a single item.
     */
    public function validateItem($specification, $candidate) {
        if(!array_key_exists("type", $specification)) {
            throw new Exception("No type specified for validation.");
        }

        switch($specification["type"]) {
        case "string":
            // Check if we can interpret it as a string
            if(is_array($candidate) ||
              !((!is_object($candidate) && settype($candidate, "string") !== false) ||
              (is_object($candidate) && method_exists($item, "__toString")))) {
                return new Crossroads_API_Helper_Validation_Result(
                    ["Invalid type for string candidate, got '".gettype($candidate)."'."]
                );
            }

            $str = (string)$candidate;

            if(self::isRequired($specification) && empty($str)) {
                return new Crossroads_API_Helper_Validation_Result(["Required string is empty"], $str);
            }

            return new Crossroads_API_Helper_Validation_Result([], $str);
        default:
            throw new Exception("Invalid type '$type' for validation.");
        }
    }
}
