<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;

require_once("Mage.php");
require_once("Mage/Catalog/Model/Product.php");
require_once("Varien/Object.php");

require_once("Model/Serializer/Abstract.php");
require_once("Model/Serializer/Groupprices.php");
require_once("Model/Serializer/Product/Abstract.php");
require_once("Model/Serializer/Product/List.php");
require_once("Model/Serializer/Product/Options.php");
require_once("Model/Serializer/Attributes.php");
require_once("Model/Serializer/Image.php");
require_once("Model/Serializer/Product/Stockinfo.php");

/**
 * @covers Crossroads_API_Model_Serializer_Product_List
 */
class Crossroads_API_Model_Serializer_Product_ListTest extends TestCase {
    public function setUp() {
        Mage::_clear();
    }

    /**
     * @dataProvider basicData
     */
    public function testBasic($showDiscount, $specialPrice) {
        $called     = false;
        $attrObj    = new stdClass;
        $smallObj   = new stdClass;
        $nullObj    = new stdClass;
        $groupObj   = new stdClass;
        $optionsObj = new stdClass;
        $qtyObj     = new stdClass;
        $backObj    = new stdClass;
        $manageObj  = new stdClass;
        $thumbObj   = new stdClass;

        $product     = $this->getMockBuilder(Mage_Catalog_Model_Product::class)
                            ->setMethods(["getShowDiscount", "getMinimalPrice", "getEntityId", "getTypeId", "getName", "getSku", "getUrlKey", "getRequestPath", "hasSpecialPrice", "getSpecialPrice", "getPrice", "getIsSalable", "getPosition", "getShortDescription"])
                            ->getMock();
        $groupprices = $this->createMock(Crossroads_API_Model_Serializer_Groupprices::class);
        $options     = $this->createMock(Crossroads_API_Model_Serializer_Product_Options::class);
        $thumbImage  = $this->createMock(Crossroads_API_Model_Serializer_Image::class);
        $smallImage  = $this->createMock(Crossroads_API_Model_Serializer_Image::class);
        $stockInfo   = $this->createMock(Crossroads_API_Model_Serializer_Product_Stockinfo::class);
        $attributes  = $this->createMock(Crossroads_API_Model_Serializer_Attributes::class);

        $groupprices->expects($this->once())
                    ->method("serializeItem")
                    ->with($this->identicalTo($product))
                    ->will($this->returnValue($groupObj));
        $options->expects($this->once())
                ->method("serializeItem")
                ->with($this->identicalTo($product))
                ->will($this->returnValue($optionsObj));
        $thumbImage->expects($this->once())
                   ->method("serializeItem")
                   ->with($this->identicalTo($product))
                   ->will($this->returnValue($thumbObj));
        $smallImage->expects($this->once())
                   ->method("serializeItem")
                   ->with($this->identicalTo($product))
                   ->will($this->returnValue($smallObj));
        $stockInfo->expects($this->once())
                  ->method("serializeItem")
                  ->with($this->identicalTo($product))
                  ->will($this->returnValue([
                      "stockQty"        => $qtyObj,
                      "stockBackorders" => $backObj,
                      "stockManage"     => $manageObj,
                  ]));
        $attributes->expects($this->once())
                   ->method("serializeItem")
                   ->with($this->identicalTo($product))
                   ->will($this->returnValue($attrObj));

        $serializer = $this->getMockBuilder(Crossroads_API_Model_Serializer_Product_List::class)
                           ->setMethods(["getMsrp"])
                           ->setConstructorArgs([[
                               "options"     => $options,
                               "thumbnail"   => $thumbImage,
                               "smallImage"  => $smallImage,
                               "stockInfo"   => $stockInfo,
                               "attributes"  => $attributes,
                               "groupPrices" => $groupprices,
                           ]])
                           ->getMock();
        $product->expects($this->once())
                ->method("getShowDiscount")
                ->will($this->returnValue($showDiscount));
        $product->expects($this->any())
                ->method("getMinimalPrice")
                ->will($this->returnValue(123));
        $product->expects($this->any())
                ->method("getEntityId")
                ->will($this->returnValue("321"));
        $product->expects($this->any())
                ->method("getSku")
                ->will($this->returnValue("ABC123"));
        $product->expects($this->any())
                ->method("getTypeId")
                ->will($this->returnValue("simple"));
        $product->expects($this->any())
                ->method("getName")
                ->will($this->returnValue("Product Name"));
        $product->expects($this->any())
                ->method("getUrlKey")
                ->will($this->returnValue("example_product"));
        $product->expects($this->any())
                ->method("getRequestPath")
                ->will($this->returnValue("category/example_product"));
        $product->expects($this->any())
                ->method("getShortDescription")
                ->will($this->returnValue("A shorter description for the product"));
        $product->expects($this->any())
                ->method("getPosition")
                ->will($this->returnValue("1234"));
        $product->expects($this->any())
                ->method("hasSpecialPrice")
                ->will($this->returnValue((bool)$specialPrice));
        $product->expects($this->any())
                ->method("getSpecialPrice")
                ->will($this->returnValue($specialPrice));

        $serializer->expects($this->once())
                   ->method("getMsrp")
                   ->with($this->identicalTo($product))
                   ->will($this->returnValue(1000));

        $expected = [
            "id"               => 321,
            "type"             => "simple",
            "name"             => "Product Name",
            "sku"              => "ABC123",
            "urlKey"           => "example_product",
            "requestPath"      => "category/example_product",
            "price"            => 123.0,
            "specialPrice"     => $specialPrice ? (double)$specialPrice : null,
            "groupPrices"      => $groupObj,
            "originalPrice"    => 0.0,
            "msrp"             => 1000,
            "discountPercent"  => $showDiscount ? 87.0 : null,
            "shortDescription" => "A shorter description for the product",
            "thumbImage"       => $thumbObj,
            "smallImage"       => $smallObj,
            "isSalable"        => false,
            "position"         => 1234,
            "options"          => $optionsObj,
            "attributes"       => $attrObj,
            "stockQty"         => $qtyObj,
            "stockBackorders"  => $backObj,
            "stockManage"      => $manageObj,
        ];

        Mage::_setEventListener(Crossroads_API_Model_Serializer_Product_List::EVENT_LIST_POST_DATA_PREPARE, function($data) use(&$called, $product, $expected) {
            $this->assertSame($product, $data["product"]);
            $this->assertInstanceOf(Varien_Object::class, $data["prepared_data"]);
            $this->assertSame($expected, $data["prepared_data"]->getData());

            $called = true;
        });

        $this->assertEquals($expected, $serializer->serializeItem($product));
        $this->assertEquals(true, $called);
    }

    public function basicData() {
        return [
            [false, false],
            [true,  false],
            [false,   100],
        ];
    }
}
