<?php

/**
 * Serializes a product with information suitable for a product list.
 *
 * Params:
 *  * attributes:  Crossroads_API_Model_Serializer_Attributes
 *  * groupPrices: Crossroads_API_Model_Serializer_Groupprices
 *  * options:     Crossroads_API_Model_Serializer_Abstract|Product_Option
 *  * smallImage:  Crossroads_API_Model_Serializer_Abstract|Image
 *  * stockInfo:   Crossroads_API_Model_Serializer_Product_StockInfo
 *  * thumbnail:   Crossroads_API_Model_Serializer_Abstract|Image
 */
class Crossroads_API_Model_Serializer_Product_List extends Crossroads_API_Model_Serializer_Product_Abstract {
    /**
     * Executed after the basic product-list associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_LIST_POST_DATA_PREPARE = "crossroads_api_product_list_post_data_prepare";

    protected $attributes  = null;
    protected $groupPrices = null;
    protected $options     = null;
    protected $smallImage  = null;
    protected $stockInfo   = null;

    public function __construct(array $params = []) {
        $this->attributes   = $params["attributes"];
        $this->groupPrices  = $params["groupPrices"];
        $this->options      = $params["options"];
        $this->smallImage   = $params["smallImage"];
        $this->stockInfo    = $params["stockInfo"];
        $this->thumbnail    = $params["thumbnail"];

        parent::__construct($params);
    }

    public function serializeItem($item) {
        $msrp             = $this->getMsrp($item);
        $discount_percent = $item->getShowDiscount() && $msrp
            ? floor((1 - ((double)$item->getMinimalPrice() / $msrp)) * 100)
            : null;

        $productData = new Varien_Object(array_merge([
            "id"               => (int) $item->getEntityId(),
            "type"             => $item->getTypeId(),
            "name"             => $item->getName(),
            "sku"              => $item->getSku(),
            "urlKey"           => $item->getUrlKey(),

            "price"            => (double)$item->getMinimalPrice(),
            "specialPrice"     => $item->hasSpecialPrice() ? ((double)$item->getSpecialPrice()) : null,
            "groupPrices"      => $this->groupPrices->serializeItem($item),
            "originalPrice"    => (double)$item->getPrice(),
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "shortDescription" => $item->getShortDescription(),
            "thumbImage"       => $this->thumbnail->serializeItem($item),
            "smallImage"       => $this->smallImage->serializeItem($item),
            "isSalable"        => $item->getIsSalable() > 0,
            "position"         => (int)$item->getPosition() ?: null,
            "options"          => $this->options->serializeItem($item),
            "attributes"       => $this->attributes->serializeItem($item),
        ], $this->stockInfo->serializeItem($item)));

        Mage::dispatchEvent(self::EVENT_LIST_POST_DATA_PREPARE, [
            "product"       => $item,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }
}
