<?php

class Crossroads_API_OrdersController extends Crossroads_API_Controller_Resource {
    /**
     * Executed for customer order list after the order list data has been prepared.
     *
     * Params:
     *  * array:    The ArrayObject to modify, this will be JSON encoded as the response
     *  * orders:   The array of order objects, in the same order as the ones in the ArrayObject
     *  * request:  The current Request object
     *  * customer: The customer for the orders
     */
    const EVENT_POST_LIST_PREPARE = "crossroads_api_order_list_post_data_prepare";
    /**
     * Executed for customer order details after the order data has been prepared.
     *
     * Params:
     *  * prepared_data: A varien object to be encoded as JSON
     *  * order:    The order object
     *  * customer: The customer for the order
     */
    const EVENT_POST_DETAIL_PREPARE = "crossroads_api_order_detail_post_data_prepare";

    /**
     * @api {get} /orders List orders
     * @apiName getAllCustomerOrders
     * @apiGroup Customer
     * @apiDescription Lists orders for the currently logged in customer.
     *
     * @apiParam {integer} [limit=20] Maximum number of items per page, maximum 100
     * @apiParam {integer} [page=1] Page number
     *
     * @apiSuccess {String}  id Order increment id
     * @apiSuccess {String}  status
     * @apiSuccess {String}  paymentMethod
     * @apiSuccess {String}  shippingMethod
     * @apiSuccess {Object}  quote
     * @apiSuccess {Object}  quote.summary
     * @apiSuccess {Double}  quote.summary.subTotal
     * @apiSuccess {Double}  quote.summary.subTotalExclTax
     * @apiSuccess {Double}  quote.summary.grandTotal
     * @apiSuccess {Double}  quote.summary.grandTotalExclTax
     * @apiSuccess {Double}  [quote.summary.discount]
     * @apiSuccess {Double}  quote.summary.tax
     * @apiSuccess {Double}  quote.summary.shippingAmount
     * @apiSuccess {Double}  quote.summary.shippingAmountExclTax
     * @apiSuccess {String}  quote.summary.quoteCurrencyCode
     * @apiSuccess {Double}  quote.summary.qty
     * @apiSuccess {Object}  [quote.summary.coupon]
     * @apiSuccess {String}  quote.summary.coupon.couponCode
     * @apiSuccess {String}  quote.summary.coupon.ruleName
     * @apiSuccess {Boolean} quote.summary.virtual
     * @apiSuccess {String}  email
     * @apiSuccess {String}  createdAt  ISO 8601 datetime, UTC
     */
    public function getAll() {
        $this->ensureLoggedIn();

        $pageSize   = min((int)$this->getRequest()->getQuery("limit", "20"), 100);
        $page       = max((int)$this->getRequest()->getQuery("page", "1"), 1);
        $helper     = Mage::helper("API/customer");
        $customer   = $helper->getCurrentCustomer();
        $addrHelper = Mage::helper("API/address");
        $orders     = Mage::getResourceModel('sales/order_collection')
            ->addFieldToSelect('*')
            ->addFieldToFilter('customer_id', $customer->getId())
            ->addFieldToFilter('state', ['in' => Mage::getSingleton('sales/order_config')->getVisibleOnFrontStates()])
            ->setOrder('created_at', 'desc');
        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $orders->getSelect()->limitPage($page, $pageSize);
        $orders->load();

        $array = new ArrayObject(array_values(array_map(function($order) use ($helper, $addrHelper) {
            return [
                "id"              => $order->getIncrementId(),
                "status"          => $order->getStatus(),
                "paymentMethod"   => $order->getPayment()->getMethod(),
                "shippingMethod"  => $order->getShippingMethod(),
                // billingAddress
                // shippingAddress
                "quote"           => [
                    // items
                    "summary" => [
                        "subTotal"              => (double)$order->getSubtotalInclTax(),
                        "subTotalExclTax"       => (double)$order->getSubtotal(),
                        // GrandTotal is always including tax
                        "grandTotal"            => (double)$order->getGrandTotal(),
                        "grandTotalExclTax"     => (double)($order->getGrandTotal() - $order->getTaxAmount()),
                        "discount"              => (double)$order->getDiscountAmount() ?: null,
                        "tax"                   => (double)$order->getTaxAmount(),
                        // taxRates
                        "shippingAmount"        => (double)($order->getShippingAmount() + $order->getShippingTaxAmount()),
                        "shippingAmountExclTax" => (double)$order->getShippingAmount(),
                        "quoteCurrencyCode"     => $order->getOrderCurrencyCode(),
                        "qty"                   => (double)$order->getTotalQtyOrdered(),
                        "coupon"                => $order->getCouponCode() ? [
                            "couponCode" => $order->getCouponCode(),
                            "ruleName"   => $order->getCouponRuleName(),
                        ] : null,
                        "virtual"               => (boolean)$order->getIsVirtual(),
                        // hasVirtualItems (boolean)
                    ],
                ],
                "email"           => $order->getCustomerEmail() ?: null,
                "createdAt"       => gmdate("Y-m-d\TH:i:s\Z", strtotime($order->getCreatedAt())),
            ];
        }, $orders->getItems())));

        Mage::dispatchEvent(self::EVENT_POST_LIST_PREPARE, [
            "array"    => $array,
            "orders"   => array_values($orders->getItems()),
            "request"  => $this->getRequest(),
            "customer" => $customer,
        ]);

        return [200, array_values($array->getArrayCopy())];
    }

    /**
     * @api {get} /orders/:id Show order
     * @apiName getCustomerOrder
     * @apiGroup Customer
     *
     * @apiParam {String}  Order increment id
     *
     * @apiSuccess {String}    id Order increment id
     * @apiSuccess {String}    status
     * @apiSuccess {String}    paymentMethod
     * @apiSuccess {String}    shippingMethod
     * @apiSuccess {Object}    billingAddress
     * @apiSuccess {String}    billingAddress.prefix
     * @apiSuccess {String}    billingAddress.firstname
     * @apiSuccess {String}    billingAddress.middlename
     * @apiSuccess {String}    billingAddress.lastname
     * @apiSuccess {String}    billingAddress.suffix
     * @apiSuccess {String}    billingAddress.company
     * @apiSuccess {String[]}  billingAddress.street
     * @apiSuccess {String}    billingAddress.postcode
     * @apiSuccess {String}    billingAddress.city
     * @apiSuccess {String}    billingAddress.regionId
     * @apiSuccess {String}    billingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}    billingAddress.telephone
     * @apiSuccess {String}    billingAddress.fax
     * @apiSuccess {bool}      billingAddress.useAsShippingAddress
     * @apiSuccess {Object}    shippingAddress
     * @apiSuccess {String}    shippingAddress.prefix
     * @apiSuccess {String}    shippingAddress.firstname
     * @apiSuccess {String}    shippingAddress.middlename
     * @apiSuccess {String}    shippingAddress.lastname
     * @apiSuccess {String}    shippingAddress.suffix
     * @apiSuccess {String}    shippingAddress.company
     * @apiSuccess {String[]}  shippingAddress.street
     * @apiSuccess {String}    shippingAddress.postcode
     * @apiSuccess {String}    shippingAddress.city
     * @apiSuccess {String}    shippingAddress.regionId
     * @apiSuccess {String}    shippingAddress.countryId (SE, NO, DK)
     * @apiSuccess {String}    shippingAddress.telephone
     * @apiSuccess {String}    shippingAddress.fax
     * @apiSuccess {Integer[]} shippingTracker  List of shipping tracking numbers, if any
     * @apiSuccess {Object}    quote
     * @apiSuccess {Object[]}  quote.items  The quote list
     * @apiSuccess {Integer}   quote.items.id
     * @apiSuccess {Object}    quote.items.product  A lightweight product object
     * @apiSuccess {Integer}   quote.items.product.id
     * @apiSuccess {String}    quote.items.product.sku
     * @apiSuccess {String}    quote.items.product.name
     * @apiSuccess {String}    quote.items.product.urlKey
     * @apiSuccess {Double}    quote.items.product.price
     * @apiSuccess {Integer}  [quote.items.product.stockQty]  Amount in stock
     * @apiSuccess {String}    quote.items.product.manufacturer
     * @apiSuccess {String}    quote.items.product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}    quote.items.qty
     * @apiSuccess {Double}    quote.items.rowTotal  Total price for the row
     * @apiSuccess {Double}    quote.items.rowTax
     * @apiSuccess {Object}   [quote.items.attributes]  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer}   quote.items.attributes[attributeId] Attribute value
     * @apiSuccess {Array}    [quote.items.options]
     * @apiSuccess {Integer}   quote.items.options.id
     * @apiSuccess {String}    quote.items.options.title
     * @apiSuccess {Boolean}   quote.items.options.useAsDefault
     * @apiSuccess {Integer}   quote.items.options.position
     * @apiSuccess {Object}    quote.items.options.value       The selected value for the complex product attribute
     * @apiSuccess {Integer}   quote.items.options.value.id
     * @apiSuccess {String}    quote.items.options.value.label
     * @apiSuccess {Boolean}   quote.items.options.value.isPercent
     * @apiSuccess {Mixed}    [quote.items.options.value.price]
     * @apiSuccess {String}   [quote.items.options.value.thumbnail]  Thumbnail of the selected option
     * @apiSuccess {Object}   [quote.items.bundleOptions]  Map of selection id => selection value(s) for bundle products
     * @apiSuccess {Integer}   quote.items.bundleOptions[selectionId]  Attribute value, or list of attribute values
     * @apiSuccess {Object[]} [quote.items.selections]
     * @apiSuccess {Integer}   quote.items.selections.id  Selection id
     * @apiSuccess {String}    quote.items.selections.title
     * @apiSuccess {Object[]}  quote.items.selections.options  List of included options
     * @apiSuccess {String}    quote.items.selections.options.name
     * @apiSuccess {Integer}   quote.items.selections.options.qty
     * @apiSuccess {Double}    quote.items.selections.options.price
     * @apiSuccess {Object}    quote.summary
     * @apiSuccess {Double}    quote.summary.subTotal
     * @apiSuccess {Double}    quote.summary.subTotalExclTax
     * @apiSuccess {Double}    quote.summary.grandTotal
     * @apiSuccess {Double}    quote.summary.grandTotalExclTax
     * @apiSuccess {Double}   [quote.summary.discount]
     * @apiSuccess {Double}    quote.summary.tax
     * @apiSuccess {Object[]}  quote.summary.taxRates
     * @apiSuccess {Double}    quote.summary.taxRates.percent
     * @apiSuccess {Double}    quote.summary.taxRates.amount
     * @apiSuccess {Double}    quote.summary.shippingAmount
     * @apiSuccess {Double}    quote.summary.shippingAmountExclTax
     * @apiSuccess {String}    quote.summary.quoteCurrencyCode
     * @apiSuccess {Double}    quote.summary.qty
     * @apiSuccess {Object}   [quote.summary.coupon]
     * @apiSuccess {String}    quote.summary.coupon.couponCode
     * @apiSuccess {String}    quote.summary.coupon.ruleName
     * @apiSuccess {Boolean}   quote.summary.virtual
     * @apiSuccess {Boolean}   quote.summary.hasVirtualItems If the quote contains any virtual items
     *                                                       (note that configurable can contain virtual)
     * @apiSuccess {String}    email
     * @apiSuccess {String}    createdAt  ISO 8601 datetime, UTC
     */
    public function getItem($id) {
        $this->ensureLoggedIn();

        $order      = Mage::getModel("sales/order")->loadByIncrementId($id);
        $addrHelper = Mage::helper("API/address");
        $helper     = Mage::helper("API/customer");

        if(!$order || !$order->getIncrementId() || !$order->getPayment() || $order->getCustomerId() != $helper->getCurrentCustomer()->getId()) {
            return [404];
        }

        $data = new Varien_Object([
            "id"              => $order->getIncrementId(),
            "status"          => $order->getStatus(),
            "paymentMethod"   => $order->getPayment()->getMethod(),
            "shippingMethod"  => $order->getShippingMethod() ? $order->getShippingMethod() : null,
            "billingAddress"  => $addrHelper->prepareOrderAddress($order->getBillingAddress(), $order),
            "shippingAddress" => $addrHelper->prepareOrderAddress($order->getShippingAddress(), $order),
            "shippingTracker" => array_values(array_map(function($t) {
                return $t->getNumber();
            }, $order->getTracksCollection()->getItems())),
            "quote"           => Mage::helper("API/Cart")->formatOrder($order),
            "email"           => $order->getCustomerEmail() ?: null,
            "createdAt"       => gmdate("Y-m-d\TH:i:s\Z", strtotime($order->getCreatedAt())),
        ]);

        Mage::dispatchEvent(self::EVENT_POST_DETAIL_PREPARE, [
            "prepared_data" => $data,
            "order"         => $order,
            "customer"      => $helper->getCurrentCustomer(),
        ]);

        return [200, $data->getData()];
    }
}
