<?php

class Crossroads_API_BrandsController extends Crossroads_API_Controller_Resource
{
    /**
     * Executed after the brand item response data is prepared, modifications to the resp
     * parameter will be sent as a response.
     *
     * Params:
     *  * brand:    Brand label string value
     *  * response: Varien object response
     */
    const EVENT_BRAND_ITEM_RESPONSE = "crossroads_api_brand_item_response";

    /**
     * @api {get} /brands Get all brands
     * @apiName getBrands
     * @apiGroup Brands
     *
     * @apiSuccess {Object[]} brands Array of the brands
     * @apiSuccess {String} brands.urlKey
     * @apiSuccess {String} brands.name
     * @apiSuccessExample Example
     * [ { "urlKey": "Presentkort", "name": "Presentkort" }, { "urlKey": "Design+House+Stockholm", "name": "Design House Stockholm" } ]
     */
    public function getAll()
    {
        $store  = Mage::app()->getStore();
        $helper = Mage::helper("API/brand");
        $model  = Mage::getModel("API/attribute");
        $attrs  = Mage::getSingleton("catalog/config")->getAttributesUsedInProductListing();

        if( ! array_key_exists("manufacturer", $attrs)) {
            Mage::throwException("Attribute manufacturer is not indexed in product flat table");
        }

        $brands = $model->fetchProductAttributeValues($attrs["manufacturer"]);

        return [200, array_values(array_map(function($c) use($helper) {
            return $helper->prepareBrand($c);
        }, $brands))];
    }

    /**
     * @api {get} /brands/:urlKey Get brand products
     * @apiName getBrand
     * @apiGroup Brands
     *
     * @apiParam {String} urlKey
     * @apiParam {Integer} [limit=20] The pagination size, max 100
     * @apiParam {Integer} [page=1] The pagination page
     *
     * @apiSuccess {Object} brand Brand
     * @apiSuccess {String} brand.name
     * @apiSuccess {String} brand.urlKey
     *
     * @apiSuccess {Object[]} products Array of the products in the brand
     * @apiSuccess {Integer}  products.id Product id
     * @apiSuccess {String}   products.type Product type
     * @apiSuccess {String}   products.sku
     * @apiSuccess {String}   products.urlKey
     * @apiSuccess {Double}   products.price
     * @apiSuccess {Double}   products.specialPrice
     * @apiSuccess {Double}   products.msrp
     * @apiSuccess {Double}   products.discountPercent
     * @apiSuccess {String}   products.shortDescription
     * @apiSuccess {Mixed}    products.manufacturer String if a manufacturer is set, null otherwise
     * @apiSuccess {String}   products.smallImage
     * @apiSuccess {Boolean}  products.isSalable
     * @apiSuccess {Object[]} products.options  Null if simple product, object if complex, can be turned off in list view through System Configuration
     * @apiSuccessExample Example
     * { "brand": { "value": "5", "label": "Presentkort" }, "products": [ { "id": 17, "type": "virtual", "name": "Biobiljett", "sku": "21092", "urlKey": "biobiljett", "price": 99, "specialPrice": null, "msrp": 110, "discountPercent": null, "shortDescription": "Lite kort test", "smallImage": "http://example.com/media/small_image.jpg", "isSalable": true, "manufacturer": "Presentkort", "options": null } ] }
     */
    public function getItem($urlKey)
    {
        // TODO: This should be more general
        $store  = Mage::app()->getStore();
        $helper = Mage::helper("API/brand");
        $model  = Mage::getModel("API/attribute");
        $attrs  = Mage::getSingleton("catalog/config")->getAttributesUsedInProductListing();

        if( ! array_key_exists("manufacturer", $attrs)) {
            Mage::throwException("Attribute manufacturer is not indexed in product flat table");
        }

        $brands = $model->fetchProductAttributeValues($attrs["manufacturer"]);

        $brand = current(array_filter($brands, function($b) use ($urlKey) {
            return $b["value"] && $b["value"] === $urlKey;
        }));

        if( ! $brand) {
            return [404];
        }

        $serializer = Mage::getModel("API/factory")->createProductListSerializer($store);
        $products   = Mage::getModel("API/collection_product_attribute")
            ->setAttributeFilter("manufacturer", $brand["id"])
            ->setLimit($this->getRequest()->getQuery("limit"))
            ->setPage($this->getRequest()->getQuery("page"))
            ->createCollection();

        $resp = new Varien_Object([
            "brand"    => $helper->prepareBrand($brand),
            "products" => $serializer->mapCollection($products),
        ]);

        Mage::dispatchEvent(self::EVENT_BRAND_ITEM_RESPONSE, [
            "brand"    => $brand["value"],
            "response" => $resp,
        ]);

        return [200, $resp->getData()];
    }
}
