<?php

/**
 * Factory which assembles serializers from store configuration.
 */
class Crossroads_API_Model_Factory extends Mage_Core_Model_Abstract {
    const CONFIG_IMAGE_LARGE_SIZE  = "API_section/images/large_image_size";
    const CONFIG_IMAGE_LARGE_FILL  = "API_section/images/large_image_fill";
    const CONFIG_IMAGE_SMALL_SIZE  = "API_section/images/small_image_size";
    const CONFIG_IMAGE_SMALL_FILL  = "API_section/images/small_image_fill";
    const CONFIG_IMAGE_THUMB_SIZE  = "API_section/images/thumbnail_image_size";
    const CONFIG_IMAGE_THUMB_FILL  = "API_section/images/thumbnail_image_fill";

    const CONFIG_LIST_STOCK_QTY    = "API_section/product_list/get_stock_qty_in_list";
    const CONFIG_LIST_OPTIONS      = "API_section/product_list/get_options_in_list";
    const CONFIG_LIST_THUMBNAIL    = "API_section/product_list/include_thumbnail";
    const CONFIG_LIST_ATTRIBUTES   = "API_section/attributes/product_list";
    const CONFIG_LIST_GROUP_PRICES = "API_section/product_list/get_group_prices_in_list";

    const CONFIG_DETAIL_ATTRIBUTES = "API_section/attributes/product_view";
    const CONFIG_DETAIL_STOCK_QTY  = "API_section/product_view/get_stock_qty";

    const CONFIG_CART_ATTRIBUTES   = "API_section/attributes/product_cart";

    const CONFIG_OPTION_ATTRIBUTES = "API_section/attributes/product_option";

    const CONFIG_CATEGORY_PRODUCT_COUNT = "API_section/categories/include_product_count";

    /**
     * @param  Mage_Core_Model_Store
     * @param  string
     * @return array
     */
    protected function getAttributeConfig($store, $config) {
        $data = $store->getConfig($config);

        return $data ? unserialize($data) : [];
    }

    public function createProductListSerializer(Mage_Core_Model_Store $store) {
        $attrs       = $this->getAttributeConfig($store, self::CONFIG_LIST_ATTRIBUTES);
        $thumbnail   = $store->getConfig(self::CONFIG_LIST_THUMBNAIL)
            ? $this->createThumbImageSerializer($store)
            : $this->createNullSerializer();
        $groupPrices = $store->getConfig(self::CONFIG_LIST_GROUP_PRICES)
            ? Mage::getModel("API/serializer_groupprices_loader", [
                "websiteId" => $store->getWebsiteId(),
                "storeId"   => $store->getId(),
            ])
            : $this->createNullSerializer();
        $stockInfo   = (bool)$store->getConfig(self::CONFIG_LIST_STOCK_QTY)
            ? Mage::getModel("API/serializer_product_stockinfo")
            : Mage::getModel("API/serializer_product_stockinfo_empty");
        $options     = $store->getConfig(self::CONFIG_LIST_OPTIONS)
            ? $this->createProductOptionSerializer($store, $groupPrices, $stockInfo)
            : $this->createNullSerializer();

        $params = [
            "attributes"  => Mage::getModel("API/serializer_attributes", $attrs),
            "groupPrices" => $groupPrices,
            "options"     => $options,
            "smallImage"  => $this->createSmallImageSerializer($store),
            "stockInfo"   => $stockInfo,
            "thumbnail"   => $thumbnail,
        ];

        return Mage::getModel("API/serializer_product_list", $params);
    }

    public function createProductDetailSerializer($store) {
        $attrs       = $this->getAttributeConfig($store, self::CONFIG_DETAIL_ATTRIBUTES);
        // Bundles share config with options
        $bundleAttrs = $this->getAttributeConfig($store, self::CONFIG_OPTION_ATTRIBUTES);
        // TODO: Shold this be separated between list and detail?
        $groupPrices = $store->getConfig(self::CONFIG_LIST_GROUP_PRICES)
            ? Mage::getModel("API/serializer_groupprices", [
                "websiteId" => $store->getWebsiteId(),
            ])
            : $this->createNullSerializer();
        $optGroup    = $store->getConfig(self::CONFIG_LIST_GROUP_PRICES)
            ? Mage::getModel("API/serializer_groupprices_loader", [
                "websiteId" => $store->getWebsiteId(),
                "storeId"   => $store->getId(),
            ])
            : $this->createNullSerializer();
        $stockInfo   = (bool)$store->getConfig(self::CONFIG_DETAIL_STOCK_QTY)
            ? Mage::getModel("API/serializer_product_stockinfo")
            : Mage::getModel("API/serializer_product_stockinfo_empty");
        $options     = $this->createProductOptionSerializer($store, $optGroup, $stockInfo);
        $largeImage  = $this->createLargeImageSerializer($store);
        $origImage   = $this->createOrigImageSerializer($store);
        $smallImage  = $this->createSmallImageSerializer($store);

        $params = [
            "attributes"    => Mage::getModel("API/serializer_attributes", $attrs),
            "bundle"        => Mage::getModel("API/serializer_product_bundle", [
                "attributes"  => Mage::getModel("API/serializer_attributes", $bundleAttrs),
                "groupPrices" => $optGroup,
                "largeImage"  => $largeImage,
                "origImage"   => $origImage,
                "smallImage"  => $smallImage,
                "stockInfo"   => $stockInfo,
            ]),
            "customOptions" => Mage::getModel("API/serializer_product_customoptions"),
            "groupPrices"   => $groupPrices,
            "largeImage"    => $largeImage,
            "mediaGallery"  => $this->createMediaGallerySerializer($store),
            "options"       => $options,
            "origImage"     => $origImage,
            "related"       => $this->createRelatedProductsSerializer($store),
            "smallImage"    => $smallImage,
            "stockInfo"     => $stockInfo,
        ];

        return Mage::getModel("API/serializer_product_detail", $params);
    }

    public function createCartItemSerializer(Mage_Core_Model_Store $store) {
        $attrs     = $this->getAttributeConfig($store, self::CONFIG_CART_ATTRIBUTES);
        $optAttrs  = $this->getAttributeConfig($store, self::CONFIG_OPTION_ATTRIBUTES);
        // TODO: Separate from detail qty?
        $stockInfo = (bool)$store->getConfig(self::CONFIG_DETAIL_STOCK_QTY)
            ? Mage::getModel("API/serializer_product_stockinfo")
            : Mage::getModel("API/serializer_product_stockinfo_empty");

        return Mage::getModel("API/serializer_product_cart", [
            "attributes"       => Mage::getModel("API/serializer_attributes", $attrs),
            "optionAttributes" => Mage::getModel("API/serializer_attributes", $optAttrs),
            "stockInfo"        => $stockInfo,
            "thumbnail"        => $this->createThumbImageSerializer($store),
        ]);
    }

    public function createWishlistSerializer(Mage_Core_Model_Store $store) {
        return Mage::getModel("API/serializer_wishlist", [
            "item" => $this->createCartItemSerializer($store),
        ]);
    }

    public function createCategorySerializer(Mage_Core_Model_Store $store, $includeImages = false) {
        $productCount = $store->getConfig(self::CONFIG_CATEGORY_PRODUCT_COUNT)
            ? Mage::getModel("API/serializer_category_productcount", [
                "collection" => Mage::getModel("API/collection_product_category")
                    ->setStore($store),
            ])
            : Mage::getModel("API/serializer_constant", [ "value" => 0 ]);

        return Mage::getModel("API/serializer_category", [
            "productCount"  => $productCount,
            "includeImages" => $includeImages,
        ]);
    }

    protected function createProductOptionSerializer(
        Mage_Core_Model_Store $store,
        Crossroads_API_Model_Serializer_Abstract $groupPrices,
        Crossroads_API_Model_Serializer_Product_Stockinfo $stockInfo) {
        $attrs       = $this->getAttributeConfig($store, self::CONFIG_OPTION_ATTRIBUTES);

        $params = [
            "attributes"    => Mage::getModel("API/serializer_attributes", $attrs),
            "customOptions" => Mage::getModel("API/serializer_product_customoptions"),
            "groupPrices"   => $groupPrices,
            "largeImage"    => $this->createLargeImageSerializer($store),
            "origImage"     => $this->createOrigImageSerializer($store),
            "smallImage"    => $this->createSmallImageSerializer($store),
            "stockInfo"     => $stockInfo,
        ];

        return Mage::getModel("API/serializer_product_options", $params);
    }

    protected function createThumbImageSerializer($store) {
        return Mage::getModel("API/serializer_image", [
            "attr" => "thumbnail",
            "size" => $store->getConfig(self::CONFIG_IMAGE_THUMB_SIZE),
            "fill" => $store->getConfig(self::CONFIG_IMAGE_THUMB_FILL),
        ]);
    }

    protected function createSmallImageSerializer($store) {
        return Mage::getModel("API/serializer_image", [
            "attr" => "small_image",
            "size" => $store->getConfig(self::CONFIG_IMAGE_SMALL_SIZE),
            "fill" => $store->getConfig(self::CONFIG_IMAGE_SMALL_FILL),
        ]);
    }

    protected function createLargeImageSerializer($store) {
        return Mage::getModel("API/serializer_image", [
            "attr" => "image",
            "size" => $store->getConfig(self::CONFIG_IMAGE_LARGE_SIZE),
            "fill" => $store->getConfig(self::CONFIG_IMAGE_LARGE_FILL),
        ]);
    }

    protected function createOrigImageSerializer() {
        return Mage::getModel("API/serializer_image_original", [
            "attr" => "image",
        ]);
    }

    protected function createMediaGallerySerializer($store) {
        return Mage::getModel("API/serializer_image_mediagallery", [
            "largeFill" => $store->getConfig(self::CONFIG_IMAGE_LARGE_FILL),
            "largeSize" => $store->getConfig(self::CONFIG_IMAGE_LARGE_SIZE),
            "smallFill" => $store->getConfig(self::CONFIG_IMAGE_SMALL_FILL),
            "smallSize" => $store->getConfig(self::CONFIG_IMAGE_SMALL_SIZE),
            "thumbFill" => $store->getConfig(self::CONFIG_IMAGE_THUMB_FILL),
            "thumbSize" => $store->getConfig(self::CONFIG_IMAGE_THUMB_SIZE),
        ]);
    }

    protected function createRelatedProductsSerializer($store) {
        return Mage::getModel("API/serializer_product_related", [
            "collection" => Mage::getModel("API/collection_product_related")
                ->setStore($store)
                ->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG, true),
            "serializer" => $this->createProductListSerializer($store),
        ]);
    }

    protected function createNullSerializer() {
        return Mage::getModel("API/serializer_constant", [ "value" => null ]);
    }
}
