<?php

/**
 * Serializes a cart item into a cart product object.
 *
 * Params:
 *  * attributes:       Crossroads_API_Model_Serializer_Attributes
 *  * optionAttributes: Crossroads_API_Model_Serializer_Attributes
 *  * stockInfo:        Crossroads_API_Model_Serializer_Product_Stockinfo
 *  * thumbnail:        Crossroads_API_Model_Serializer_Image
 */
class Crossroads_API_Model_Serializer_Product_Cart extends Crossroads_API_Model_Serializer_Product_Abstract {
    /**
     * Executed after the basic product-cart associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * item           The cart item
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_CART_POST_DATA_PREPARE = "crossroads_api_product_cart_post_data_prepare";

    protected $attributes = null;
    protected $stockInfo  = null;
    protected $thumbnail  = null;
    protected $optionAttrs = null;

    public function __construct(array $params = []) {
        $this->attributes    = $params["attributes"];
        $this->optionAttrs   = $params["optionAttributes"];
        $this->stockInfo     = $params["stockInfo"];
        $this->thumbnail     = $params["thumbnail"];

        parent::__construct($params);
    }

    /**
     * @param  Cart item
     */
    public function serializeItem($item) {
        $attributes    = null;
        $bundleOptions = null;
        $customOptions = null;
        $product       = $item->getProduct();
        $options       = $product->getTypeInstance(true)->getOrderOptions($product);
        $rowTotal      = $item->getRowTotal()
                       + $item->getTaxAmount()
                       + $item->getHiddenTaxAmount()
                       - $item->getDiscountAmount();
        $simple        = $item->getOptionByCode('simple_product')
            ? $item->getOptionByCode("simple_product")->getProduct()
            : $product;

        if( ! empty($options["info_buyRequest"]) &&
            is_array($options["info_buyRequest"])) {
            if(array_key_exists("super_attribute", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["super_attribute"])) {
                $attributes = [];

                foreach($options["info_buyRequest"]["super_attribute"] as $k => $v) {
                    $attributes["$k"] = (int)$v;
                }
            }

            if(array_key_exists("bundle_option", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["bundle_option"])) {
                $bundleOptions = [];

                foreach($options["info_buyRequest"]["bundle_option"] as $k => $v) {
                    $bundleOptions["$k"] = is_array($v) ? array_values(array_map(function($i) { return (int)$i; }, $v)): (int)$v;
                }
            }

            if(array_key_exists("options", $options["info_buyRequest"]) &&
                is_array($options["info_buyRequest"]["options"])) {
                $customOptions = [];

                foreach($options["info_buyRequest"]["options"] as $k => $v) {
                    $customOptions["$k"] = $v;
                }
            }
        }

        $productData = new Varien_Object([
            "id"        => (int)$item->getId(),
            "product"   => array_merge([
                "id"              => (int)$product->getId(),
                "sku"             => $product->getSku(),
                "name"            => $product->getName(),
                "urlKey"          => $product->getUrlPath(),
                "price"           => (double)$item->getPriceInclTax(),
                "isSalable"       => $product->getIsSalable() > 0,
                "thumbnail"       => $this->thumbnail->serializeItem($product),
                "attributes"      => $this->attributes->serializeItem($product),
            ], $this->stockInfo->serializeItem($simple)),
            "qty"           => (double)($item->getQty() ?: $item->getQtyOrdered()),
            "rowTotal"      => (double)$rowTotal,
            "rowTax"        => (double)$item->getTaxAmount(),
            "attributes"    => $attributes,
            "options"       => $this->prepareOptionsCart($product, $options),
            "customOptions" => $customOptions,
            "bundleOptions" => $bundleOptions,
            "selections"    => $this->prepareSelectionsCart($product, $options),
        ]);

        Mage::dispatchEvent(self::EVENT_CART_POST_DATA_PREPARE, [
            "item"          => $item,
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    // TODO: Generalize this, probably needs a wrapper around the product item, or some kind of cart-reimplementation. Preferably we want to reuse stuff from Crossroads_API_Model_Serializer_Product_Options without having to reimplement it
    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareOptionsCart($product, $options) {
        if($product->getTypeId() !== self::PRODUCT_CONFIGURABLE) {
            return null;
        }

        if (empty($options["info_buyRequest"]) ||
            !is_array($options["info_buyRequest"]["super_attribute"])) {
            return null;
        }

        $attrs      = $options["info_buyRequest"]["super_attribute"];
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = $instance->getUsedProducts(null, $product);

        // Specific options with specific set values
        return array_map(function($option) use($attrs, $children) {
            $value = null;
            $child = null;

            // Attempt to find the specific child-product for the selected attribute value
            foreach($children as $c) {
                if((int)$c->getData($option["attribute_code"]) === (int)$attrs[$option["attribute_id"]]) {
                    $child = $c;

                    break;
                }
            }

            // Find the selected attribute-value
            foreach($option["values"] as $v) {
                if((int)$attrs[$option["attribute_id"]] === (int)$v["value_index"]) {
                    $value = $v;

                    break;
                }
            }

            if($value && ! $child) {
                throw new Exception(sprintf("Cart attribute %s: %s failed to obtain a child product.", $option["attribute_code"], $v["value_index"]));
            }

            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                // There might be a case where we cannot find the selected value
                "value"        => $value ? array_merge([
                    "id"         => (int)$value["value_index"],
                    "sku"        => $child->getSku(),
                    "label"      => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                    "isPercent"  => $value["is_percent"] > 0,
                    "isInStock"  => $child->isInStock() > 0,
                    "isSalable"  => $child->getIsSalable() > 0,
                    "thumbnail"  => $this->thumbnail->serializeItem($child),
                    "price"      => (double)$child->getPrice(),
                    "msrp"       => $child->getMsrp() ? (double)$child->getMsrp() : null,
                    "attributes" => $this->optionAttrs->serializeItem($child),
                ], $this->stockInfo->serializeItem($child)) : null
            ];
        }, $rawOptions);
    }

    // TODO: Generalize this, probably needs a wrapper around the product item, or some kind of cart-reimplementation.
    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareSelectionsCart($product, $options) {
        if($product->getTypeId() !== self::PRODUCT_BUNDLE) {
            return null;
        }

        if (empty($options["info_buyRequest"]) ||
            !is_array($options["bundle_options"])) {
            return null;
        }

        $selections = [];

        foreach($options["bundle_options"] as $selectionId => $selection) {
            $selections[] = [
                "id"      => (int)$selectionId,
                "title"   => $selection["label"],
                "options" => array_values(array_map(function($productData) {
                    return [
                        "name"  => $productData["title"],
                        "qty"   => (int)$productData["qty"],
                        "price" => (double)$productData["price"],
                    ];
                }, $selection["value"])),
            ];
        }

        return $selections;
    }
}
