<?php

class Crossroads_API_WishlistController extends Crossroads_API_Controller_Resource
{
    protected function getWishlist() {
        if (!Mage::getSingleton('customer/session')->isLoggedIn()) {
            throw Crossroads_API_ResponseException::create(400, "Customer is not logged in", null, 3000);
        }

        $customerId = Mage::getSingleton("customer/session")->getCustomer()->getId();

        return Mage::getModel("wishlist/wishlist")->loadByCustomer($customerId, true);
    }

    protected function getCartRequestItemsData() {
        $data = $this->requestData();

        if( ! is_array($data) || ! array_key_exists("items", $data)) {
            throw Crossroads_API_ResponseException::create(400, "Request body must have an items array", null, 2011);
        }

        $stringKeys = array_filter(array_keys($data["items"]), "is_string");

        if( ! empty($stringKeys)) {
            throw Crossroads_API_ResponseException::create(400, "Request body must have an items array", null, 2011);
        }

        return $data["items"];
    }

    /**
     * @apiDefine wishlistItems
     *
     * @apiSuccess {Object[]}  items  The quote list
     * @apiSuccess {Integer}   items.id
     * @apiSuccess {Object}    items.product  A lightweight product object
     * @apiSuccess {Integer}   items.product.id
     * @apiSuccess {String}    items.product.sku
     * @apiSuccess {String}    items.product.name
     * @apiSuccess {String}    items.product.urlKey
     * @apiSuccess {Double}    items.product.price
     * @apiSuccess {Integer}  [items.product.stockQty]  Amount in stock
     * @apiSuccess {String}    items.product.manufacturer
     * @apiSuccess {String}    items.product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}    items.qty
     * @apiSuccess {Double}    items.rowTotal  Total price for the row
     * @apiSuccess {Double}    items.rowTax
     * @apiSuccess {Object}   [items.attributes]  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer}   items.attributes[attributeId] Attribute value
     * @apiSuccess {Array}    [items.options]
     * @apiSuccess {Integer}   items.options.id
     * @apiSuccess {String}    items.options.title
     * @apiSuccess {Boolean}   items.options.useAsDefault
     * @apiSuccess {Integer}   items.options.position
     * @apiSuccess {Object}    items.options.value       The selected value for the complex product attribute
     * @apiSuccess {Integer}   items.options.value.id
     * @apiSuccess {String}    items.options.value.label
     * @apiSuccess {Boolean}   items.options.value.isPercent
     * @apiSuccess {Mixed}    [items.options.value.price]
     * @apiSuccess {String}   [items.options.value.thumbnail]  Thumbnail of the selected option
     * @apiSuccess {Object}   [items.bundleOptions]  Map of selection id => selection value(s) for bundle products
     * @apiSuccess {Integer}   items.bundleOptions[selectionId]  Attribute value, or list of attribute values
     * @apiSuccess {Object[]} [items.selections]
     * @apiSuccess {Integer}   items.selections.id  Selection id
     * @apiSuccess {String}    items.selections.title
     * @apiSuccess {Object[]}  items.selections.options  List of included options
     * @apiSuccess {String}    items.selections.options.name
     * @apiSuccess {Integer}   items.selections.options.qty
     * @apiSuccess {Double}    items.selections.options.price
     * @apiSuccess {String}    shareCode Code used for sharing the wishlist
     */

    /**
     * @api {post} /wishlist Add product to wishlist
     * @apiHeader {String} Content-Type application/json
     * @apiName createWishlistItem
     * @apiGroup Wishlist
     *
     * @apiParam {Integer}  id                          The product id
     * @apiParam {Number}   qty                         Number of items to add
     *                                                  in the case of multiselect selections
     *
     * @apiUse wishlistItems
     */
    public function createItem()
    {
        $params = $this->requestData();
        $id     = $params["id"];

        if (!$id) {
            return [404];
        }

        $requestObject = new Varien_Object([]);

        $wishlist = $this->getWishlist();

        $wishlist->addNewItem($id, $requestObject);

        $wishlist->save();

        return $this->getAll();
    }

    /**
     * @api {put} /wishlist/:id Update product wishlist item
     * @apiHeader {String} Content-Type application/json
     * @apiName replaceWishlistItem
     * @apiGroup Wishlist
     * @apiDescription Returns the contents of the whole cart despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number}   id                          ID of wishlist item
     * @apiParam {Object}   product
     * @apiParam {Integer}  product.id
     * @apiParam {Number}   qty                         New item quantity
     * @apiParam {Object}  [attributes]                 Map of attribute id => attribute value for configurable products
     * @apiParam {Integer}  attributes[attributeId]     Attribute value
     * @apiParam {Object}  [bundleOptions]              Map of selection id => option(s) for bundle products
     * @apiParam {Integer}  bundleOptions[selectionId]  Selection option id or list of selection option ids
     *                                                  in the case of multiselect selections
     *
     * @apiUse wishlistItems
     */
    public function replaceItem($id) {
        if (!$id) {
            return [404];
        }

        $wishlist = $this->getWishlist();

        $wishlist->updateItem($id, $this->requestData());

        $wishlist->save();

        return $this->getAll();
    }

    /**
     * @api {delete} /wishlist/:id Remove product from wishlist
     * @apiName deleteWishlistItem
     * @apiGroup Wishlist
     * @apiDescription Returns the contents of the whole wishlist despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number} id  ID of wishlist item
     *
     * @apiUse wishlistItems
     */
    public function deleteItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $wishlist = $this->getWishlist();
        $item = $wishlist->getItem($id);

        if( ! $item) {
            return [404];
        }

        $item->delete();

        $wishlist->save();

        return $this->getAll();
    }

    /**
     * @api {get} /wishlist/:id Get product from wishlist
     * @apiName getWishlistItem
     * @apiGroup Wishlist
     *
     * @apiParam {Number} id  ID of wishlist item
     *
     * @apiSuccess {Integer} id
     * @apiSuccess {Object}  product  A lightweight product object
     * @apiSuccess {Integer} product.id
     * @apiSuccess {String}  product.sku
     * @apiSuccess {String}  product.name
     * @apiSuccess {String}  product.urlKey
     * @apiSuccess {Double}  product.price
     * @apiSuccess {mixed}   product.stockQty               Amount in stock, null if information not available
     * @apiSuccess {String}  product.manufacturer
     * @apiSuccess {String}  product.thumbnail       Complete URL to a thumbnail for the product
     * @apiSuccess {Double}  qty
     * @apiSuccess {Double}  rowTotal                Total price for the row
     * @apiSuccess {Object}  attributes              Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer} attributes[attributeId] Attribute value
     * @apiSuccess {Object}   [bundleOptions]               Map of selection id => selection value(s) for bundle products
     * @apiSuccess {Integer}   bundleOptions[selectionId]  Attribute value, or list of attribute values
     */
    public function getItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $wishlist = $this->getWishlist();
        $item = $wishlist->getItem($id);
        if($item) {
            return [200, Mage::helper("API/Product")->prepareCartProduct($item)];
        }

        return [404];
    }

    /**
     * @api {get} /wishlist Get wishlist
     * @apiName getWishlistItems
     * @apiGroup Wishlist
     *
     * @apiUse wishlistItems
     */
    public function getAll()
    {
        $wishlist = $this->getWishlist();

        return [200, Mage::helper('API/Wishlist')->formatWishlist($wishlist)];
    }

    /**
     * @api {delete} /wishlist Empty wishlist
     * @apiName emptyWishlist
     * @apiGroup Wishlist
     *
     * @apiSuccessExample {json} Empty-wishlist Response:
     * { "items": [], "summary": { "subTotal": 0, "subTotalExclTax": 0, "grandTotal": 0, "grandTotalExclTax": 0, "tax": null, "discount": null, "shippingAmount": 0, "quoteCurrencyCode": "SEK", "qty": null, "couponCode": null } }
     * @apiUse wishlistItems
     */
    public function deleteAll()
    {
        $wishlist = $this->getWishlist();
        $wishlist->delete();
        return $this->getAll();
    }
}
