<?php

class Crossroads_API_BrandsController extends Crossroads_API_Controller_Resource
{
    /**
     * Executed after the brand item response data is prepared, modifications to the resp
     * parameter will be sent as a response.
     *
     * Params:
     *  * brand:    Brand label string value
     *  * response: Varien object response
     */
    const EVENT_BRAND_ITEM_RESPONSE = "crossroads_api_brand_item_response";

    /**
     * @api {get} /brands Get all brands
     * @apiName getBrands
     * @apiGroup Brands
     *
     * @apiSuccess {Object[]} brands Array of the brands
     * @apiSuccess {String} brands.urlKey
     * @apiSuccess {String} brands.name
     * @apiSuccessExample Example
     * [ { "urlKey": "Presentkort", "name": "Presentkort" }, { "urlKey": "Design+House+Stockholm", "name": "Design House Stockholm" } ]
     */
    public function getAll()
    {
        $helper = Mage::helper("API/Brand");
        $brands = $helper->fetchAllBrands(Mage::app()->getStore()->getStoreId());

        return [200, array_values(array_map(function($c) use($helper) {
            return $helper->prepareBrand($c);
        }, $brands))];
    }

    /**
     * @api {get} /brands/:urlKey Get brand products
     * @apiName getBrand
     * @apiGroup Brands
     *
     * @apiParam {String} urlKey
     * @apiParam {Integer} [limit=20] The pagination size, max 100
     * @apiParam {Integer} [page=1] The pagination page
     *
     * @apiSuccess {Object} brand Brand
     * @apiSuccess {String} brand.name
     * @apiSuccess {String} brand.urlKey
     *
     * @apiSuccess {Object[]} products Array of the products in the brand
     * @apiSuccess {Integer}  products.id Product id
     * @apiSuccess {String}   products.type Product type
     * @apiSuccess {String}   products.sku
     * @apiSuccess {String}   products.urlKey
     * @apiSuccess {Double}   products.price
     * @apiSuccess {Double}   products.specialPrice
     * @apiSuccess {Double}   products.msrp
     * @apiSuccess {Double}   products.discountPercent
     * @apiSuccess {String}   products.shortDescription
     * @apiSuccess {Mixed}    products.manufacturer String if a manufacturer is set, null otherwise
     * @apiSuccess {String}   products.smallImage
     * @apiSuccess {Boolean}  products.isSalable
     * @apiSuccess {Object[]} products.options  Null if simple product, object if complex, can be turned off in list view through System Configuration
     * @apiSuccessExample Example
     * { "brand": { "value": "5", "label": "Presentkort" }, "products": [ { "id": 17, "type": "virtual", "name": "Biobiljett", "sku": "21092", "urlKey": "biobiljett", "price": 99, "specialPrice": null, "msrp": 110, "discountPercent": null, "shortDescription": "Lite kort test", "smallImage": "http://example.com/media/small_image.jpg", "isSalable": true, "manufacturer": "Presentkort", "options": null } ] }
     */
    public function getItem($urlKey)
    {
        $store_id = Mage::app()->getStore()->getStoreId();
        $pageSize = min((int)$this->getRequest()->getQuery("limit", "20"), 100);
        $page     = max((int)$this->getRequest()->getQuery("page", "1"), 1);

        $brandHelper   = Mage::helper("API/brand");
        $productHelper = Mage::helper("API/product");

        // TODO: This should be more general
        $brand_attribute = Mage::getModel('eav/config')->getAttribute('catalog_product', 'manufacturer');
        $brands          = $brand_attribute->getSource()->getAllOptions(true, true);

        $brand = current(array_filter($brands, function($b) use ($urlKey) {
            return $b["value"] && $b["label"] === $urlKey;
        }));

        if( ! $brand) {
            return [404];
        }

        $products = Mage::getModel('catalog/product')
            ->getCollection()
            ->setStoreId($store_id)
            ->addAttributeToSelect('*')
            ->addMinimalPrice()
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            )
            ->addAttributeToFilter('manufacturer', $brand["value"]);
        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $products->getSelect()->limitPage($page, $pageSize);
        $products->load();

        if( ! $products || !$brand) {
            return [404];
        }

        $resp = new Varien_Object([
            "brand"    => [
                "urlKey" => rawurlencode($brand["label"]),
                "name"   => $brand["label"],
            ],
            "products" => array_values(array_map(function($p) use($productHelper) {
                return $productHelper->prepareListProduct($p);
            }, $products->getItems()))
        ]);

        Mage::dispatchEvent(self::EVENT_BRAND_ITEM_RESPONSE, [
            "brand"    => $brand["label"],
            "response" => $resp,
        ]);

        return [200, $resp->getData()];
    }
}
