<?php

class Crossroads_API_Helper_Customer {
    /**
     * Executed after the basic customer associated array has been prepared. This event allows
     * modification of the returned customer associated array.
     *
     * Params:
     *  * customer        The customer object
     *  * group           The customer group object
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     */
    const EVENT_CUSTOMER_POST_DATA_PREPARE = "crossroads_api_customer_post_data_prepare";

    protected $allowRegister = false;

    protected $customerAttributes = null;

    public function __construct() {
        $store = Mage::app()->getStore();

        $this->allowRegister      = (boolean)$store->getConfig("API_section/customer_accounts/allow_register");
        $this->customerAttributes = unserialize($store->getConfig("API_section/customer_accounts/attributes"));
    }

    /**
     * Returns true if the current session has a logged in customer.
     *
     * @return boolean
     */
    public function isLoggedIn() {
        return (boolean)Mage::getSingleton("customer/session")->isLoggedIn();
    }

    public function allowRegister() {
        return $this->allowRegister;
    }

    public function getCurrentCustomer() {
        return Mage::getSingleton('customer/session')->getCustomer();
    }

    /**
     * Creates a new customer
     *
     * @param  array  Associated array of data
     * @return Customer
     */
    public function createCustomer($data) {
        if(empty($data["email"])) {
            throw Crossroads_API_ResponseException::create(400, "Customer is missing email", null, 3002);
        }

        if(empty($data["password"])) {
            throw Crossroads_API_ResponseException::create(400, "Customer is missing password", null, 3002);
        }

        if( ! Zend_Validate::is($data["email"], "EmailAddress")) {
            throw Crossroads_API_ResponseException::create(400, "New customer email is invalid", null, 3024);
        }

        unset($data["createdAt"]);
        unset($data["updatedAt"]);

        $app      = Mage::app();
        $store    = $app->getStore();
        $website  = $app->getWebsite();
        $customer = Mage::getModel("customer/customer");
        $groupId  = $store->getConfig("API_section/customer_accounts/default_customer_group_id");

        $attributeData = array_key_exists("attributes", $data) && is_array($data["attributes"]) ? $data["attributes"] : [];

        if(array_key_exists("defaultPaymentMethod", $data)) {
            if(Mage::helper("API/payment")->isValidPaymentMethod($data["defaultPaymentMethod"])) {
                $customer->setApiDefaultPaymentMethod($data["defaultPaymentMethod"]);
            }
            else {
                $customer->unsApiDefaultPaymentMethod();
            }
        }

        unset($data["attributes"]);
        unset($data["defaultPaymentMethod"]);

        $customer->addData($data);

        Mage::helper("API/attributes")->updateEntityAttributes($customer, $this->customerAttributes, $attributeData);

        $customer->setWebsiteId($website->getId());
        $customer->setStore($store);
        $customer->setEmail($data["email"]);
        $customer->setPassword($data["password"]);
        $customer->setData("can_subscribe", 1);

        if($groupId) {
            $customer->setData("group_id", $groupId);
        }

        $customer->save();

        Mage::dispatchEvent("customer_register_success", ["customer" => $customer]);

        return $customer;
    }

    public function loginCustomer($email, $password) {
        $session  = Mage::getSingleton("customer/session");
        $customer = Mage::getModel("customer/customer")->setStore(Mage::app()->getStore())->loadByEmail($email);

        $session->login($email, $password);
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        return $customer;
    }

    public function updateCustomer($data) {
        $data      = Mage::helper("API")->filterInput($data);
        $customer  = Mage::getSingleton('customer/session')->getCustomer();
        $quote     = Mage::getSingleton("checkout/session")->getQuote();
        $savequote = false;

        $attributeData = array_key_exists("attributes", $data) && is_array($data["attributes"]) ? $data["attributes"] : [];

        if(array_key_exists("email", $data)) {
            if( ! is_scalar($data["email"]) || strlen(trim((string)$data["email"])) === 0) {
                throw Crossroads_API_ResponseException::create(400, "Customer is missing email", null, 3022);
            }

            if( ! Zend_Validate::is($data["email"], "EmailAddress")) {
                throw Crossroads_API_ResponseException::create(400, "New customer email is invalid", null, 3023);
            }
        }

        if(array_key_exists("defaultPaymentMethod", $data)) {
            if(Mage::helper("API/payment")->isValidPaymentMethod($data["defaultPaymentMethod"])) {
                if($quote &&
                   $quote->getId() &&
                   $quote->getPayment()->getMethod() != $data["defaultPaymentMethod"] &&
                   Mage::helper("API/payment")->isValidPaymentMethod($data["defaultPaymentMethod"], $quote)) {
                    $quote->getPayment()->importData([
                        "method" => $data["defaultPaymentMethod"],
                    ]);

                    $savequote = true;

                    $quote->setTotalsCollectedFlag(false);
                }

                $customer->setApiDefaultPaymentMethod($data["defaultPaymentMethod"]);
            }
            else {
                $customer->unsApiDefaultPaymentMethod();
            }
        }

        if(array_key_exists("defaultShippingMethod", $data)) {
            if(Mage::helper("API/shipping")->isValidShippingMethod($customer, $data["defaultShippingMethod"])) {
                if($quote &&
                   $quote->getId() &&
                   $quote->getShippingAddress()->getShippingMethod() != $data["defaultShippingMethod"] &&
                   !$quote->isVirtual() &&
                   $quote->getShippingAddress()->getShippingRateByCode($data["defaultShippingMethod"])) {
                    $quote->getShippingAddress()
                        ->setShippingMethod($data["defaultShippingMethod"])
                        ->setCollectShippingRates(true);

                    $quote->setTotalsCollectedFlag(false);

                    $savequote = true;
                }

                $customer->setApiDefaultShippingMethod($data["defaultShippingMethod"]);
            }
            else {
                $customer->unsApiDefaultShippingMethod();
            }
        }

        unset($data["attributes"]);
        unset($data["defaultPaymentMethod"]);
        unset($data["defaultShippingMethod"]);

        if($savequote) {
            $quote->collectTotals();
            $quote->save();
        }

        $customer->addData($data);

        Mage::helper("API/attributes")->updateEntityAttributes($customer, $this->customerAttributes, $attributeData);

        if(array_key_exists("password", $data)) {
            $customer->setPassword($data["password"]);
        }

        $customer->save();

        return $customer;
    }

    public function prepareCustomer(Mage_Customer_Model_Customer $customer) {
        $groupId = $customer->getGroupId();
        $group   = Mage::getModel("customer/group")->load($groupId);
        $data    = new Varien_Object([
            "id"         => (int)$customer->getId(),
            "email"      => $customer->getEmail(),
            "prefix"     => $customer->getPrefix(),
            "firstname"  => $customer->getFirstname(),
            "middlename" => $customer->getMiddlename(),
            "lastname"   => $customer->getLastname(),
            "suffix"     => $customer->getSuffix(),
            "VATno"      => $customer->getTaxvat(),
            "dob"        => $customer->getDob() ? strstr($customer->getDob(), " ", true) : null,
            "group"      => $group && $group->hasData("customer_group_id") ? [
                "id"   => (int)$group->getCustomerGroupId(),
                "code" => $group->getCustomerGroupCode()
            ] : null,
            "attributes" => Mage::helper("API/attributes")->getEntityAttributes($customer, $this->customerAttributes),
            "defaultPaymentMethod" => $customer->getApiDefaultPaymentMethod() ?: null,
            "defaultShippingMethod" => $customer->getApiDefaultShippingMethod() ?: null,
            "isConfirmationRequired" => $customer->getConfirmation() && $customer->isConfirmationRequired(),
            "createdAt"  => gmdate("Y-m-d\TH:i:s\Z", strtotime($customer->getCreatedAt())),
            "updatedAt"  => gmdate("Y-m-d\TH:i:s\Z", strtotime($customer->getUpdatedAt())),
        ]);

        Mage::dispatchEvent(self::EVENT_CUSTOMER_POST_DATA_PREPARE, [
            "customer"      => $customer,
            "group"         => $group,
            "prepared_data" => $data
        ]);

        return $data->getData();
    }

    public function isEmailAvailable($email, $websiteId = null) {
        $res  = Mage::getResourceModel("core/resource");
        $conn = $res->getReadConnection();

        $query = $conn->select()
            ->from($res->getTable("customer/entity"), [ "count" => "COUNT(1)" ])
            ->where("email = ?", $email);

        if($websiteId) {
            $query->where("website_id = ?", $websiteId);
        }

        $count = $conn->fetchOne($query);

        return $count < 1;
    }
}
