<?php

class Crossroads_API_Helper_Category {
    /**
     * Executed after the basic category associated array has been prepared. This event allows
     * modification of the returned category associated array.
     *
     * Params:
     *  * category        The category object
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     *  * include_images  Boolean, if the user (or action) requests that images should be included
     */
    const EVENT_CATEGORY_POST_DATA_PREPARE = "crossroads_api_category_post_data_prepare";

    public function __construct() {
        $store = Mage::app()->getStore();

        $this->includeCount = $store->getConfig("API_section/categories/include_product_count");
    }

    /**
     * Prepares a category, returning data suitable for JSON-encoding.
     *
     * @param  Category
     * @param  Set to true to include images in the category
     * @param  Product collection already prepared for the category, if any
     * @return Array Associative array
     */
    public function prepareCategory($category, $includeImages, $productCollection = null)
    {
        $catData = new Varien_Object([
            "id"              => (int)$category->getEntityId(),
            "name"            => $category->getName(),
            "description"     => $category->getDescription(),
            "metaDescription" => $category->getMetaDescription(),
            "urlKey"          => $category->getUrlKey(),
            "url"             => $category->getUrlPath(),
            "position"        => (int)$category->getPosition(),
            "level"           => (int)$category->getLevel(),
            "isAnchor"        => $category->getIsAnchor() > 0,
            "path"            => $category->getPath(),
            "parentId"        => (int)$category->getParentId(),
            "metaTitle"       => $category->getMetaTitle(),
            "includeInMenu"   => $category->getIncludeInMenu() > 0,
            "productCount"    => $this->includeCount ? ($productCollection ? $productCollection->getSize() : $this->countCategoryProducts($category)) : null
        ]);

        Mage::dispatchEvent(self::EVENT_CATEGORY_POST_DATA_PREPARE, [
            "category"       => $category,
            "prepared_data"  => $catData,
            "include_images" => $includeImages
        ]);

        return $catData->getData();
    }

    /**
     * Create a product collection containing the products for the given category, does not fetch
     * the collection or applies any limits or offsets.
     *
     * @param  Mage_Catalog_Model_Category
     * @return Varien_Db_Collection
     */
    public function createProductCollection($category) {
        return Mage::getModel('catalog/product')
            ->getCollection()
            ->addCategoryFilter($category)
            ->addAttributeToSelect("*")
            ->addMinimalPrice()
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            )
            ->addAttributeToSort("position", "asc");
    }

    protected function countCategoryProducts($category) {
        return $this->createProductCollection($category)->getSize();
    }
}