<?php

class Crossroads_API_Helper_Address {
    /**
     * Executed after the basic address associated array has been prepared. This event allows
     * modification of the returned address associated array.
     *
     * Params:
     *  * address         The address object
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     *  * quote           The quote object associated with the addres
     */
    const EVENT_QUOTE_ADDRESS_POST_DATA_PREPARE = "crossroads_api_quote_address_post_data_prepare";

    /**
     * Executed after the basic address associated array has been prepared. This event allows
     * modification of the returned address associated array.
     *
     * Params:
     *  * address         The address object
     *  * prepared_data   The associated array as a varien object, note that the data is in
     *                    camel-case, so use setData and getData to modify.
     *  * customer        The quote object associated with the addres
     */
    const EVENT_CUSTOMER_ADDRESS_POST_DATA_PREPARE = "crossroads_api_quote_address_post_data_prepare";

    protected function prepareBase($address) {
        return new Varien_Object([
            "id"         => (int)$address->getId(),
            "prefix"     => $address->getPrefix(),
            "firstname"  => $address->getFirstname(),
            "middlename" => $address->getMiddlename(),
            "lastname"   => $address->getLastname(),
            "suffix"     => $address->getSuffix(),
            "company"    => $address->getCompany(),
            "street"     => $address->getStreet(),
            "postcode"   => $address->getPostcode(),
            "city"       => $address->getCity(),
            "regionId"   => $address->getRegionId(),
            "countryId"  => $address->getCountryId(),
            "telephone"  => $address->getTelephone(),
            "fax"        => $address->getFax(),
        ]);
    }

    /**
     * Returns true if the address is empty.
     *
     * @return boolean
     */
    public function addressIsEmpty(Mage_Customer_Model_Address_Abstract $address, $skipCountryRegion = true) {
        return ( ! $address->getId()) ||
               ! trim($address->getFirstname()) &&
               ! trim($address->getLastname()) &&
               ! trim($address->getCompany()) &&
               ! trim(implode("", array_filter(array_map("trim", $address->getStreet())))) &&
               ! trim($address->getPostcode()) &&
               ! trim($address->getCity()) &&
               ! trim($address->getTelephone()) &&
               // We skip countryId and regionId since it might be some wrong default value.
               ($skipCountryRegion || ! trim($address->getRegionId()) && ! trim($address->getCountryId()));
    }

    /**
     * Transforms an address object into an address hashmap for sending to client.
     *
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Mage_Sales_Model_Quote
     * @return Array<string>
     */
    public function prepareCustomerAddress(Mage_Customer_Model_Address $address, Mage_Customer_Model_Customer $customer) {
        $data = $this->prepareBase($address);

        $data->setData("isDefaultBillingAddress", $customer->getData("default_billing") == $address->getId());
        $data->setData("isDefaultShippingAddress", $customer->getData("default_shipping") == $address->getId());

        Mage::dispatchEvent(self::EVENT_CUSTOMER_ADDRESS_POST_DATA_PREPARE, [
            "address"       => $address,
            "customer"      => $customer,
            "prepared_data" => $data
        ]);

        return $data->getData();
    }

    /**
     * Transforms an address object into an address hashmap for sending to client.
     *
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Mage_Sales_Model_Quote
     * @return Array<string>
     */
    public function prepareQuoteAddress(Mage_Sales_Model_Quote_Address $address, Mage_Sales_Model_Quote $quote) {
        $data = $this->prepareBase($address);

        // We are not interested in id property for checkout
        $data->unsetData("id");

        if($address->getAddressType() === Mage_Sales_Model_Quote_Address::TYPE_BILLING) {
            $data->setData("useAsShippingAddress", (bool)$quote->getShippingAddress()->getSameAsBilling());
        }

        Mage::dispatchEvent(self::EVENT_QUOTE_ADDRESS_POST_DATA_PREPARE, [
            "address"       => $address,
            "quote"         => $quote,
            "prepared_data" => $data
        ]);

        return $data->getData();
    }
}
