<?php

class Crossroads_API_CustomerController extends Crossroads_API_Controller_Resource {
    /**
     * Event fired after customer addresses are listed.
     *
     * Parameters:
     *  * customer: The customer
     *  * addresses: ArrayObject to be sent to client.
     */
    const EVENT_CUSTOMER_ADDRESSES_LIST_POST_DATA_PREPARE = "crossroads_api_customer_adddresses_list_post_data_prepare";

    protected function loadAddressData($address, $data) {
        $data = Mage::helper("API")->filterInput($data);

        if(array_key_exists("prefix", $data))     { $address->setPrefix($data["prefix"]); }
        if(array_key_exists("firstname", $data))  { $address->setFirstname($data["firstname"]); }
        if(array_key_exists("middlename", $data)) { $address->setMiddlename($data["middlename"]); }
        if(array_key_exists("lastname", $data))   { $address->setLastname($data["lastname"]); }
        if(array_key_exists("suffix", $data))     { $address->setSuffix($data["suffix"]); }
        if(array_key_exists("company", $data))    { $address->setCompany($data["company"]); }
        if(array_key_exists("street", $data))     { $address->setStreet($data["street"]); }
        if(array_key_exists("city", $data))       { $address->setCity($data["city"]); }
        if(array_key_exists("regionId", $data))   { $address->setRegionId($data["regionId"]); }
        if(array_key_exists("postcode", $data))   { $address->setPostcode($data["postcode"]); }
        if(array_key_exists("countryId", $data))  { $address->setCountryId($data["countryId"]); }
        if(array_key_exists("telephone", $data))  { $address->setTelephone($data["telephone"]); }
        if(array_key_exists("fax", $data))        { $address->setFax($data["fax"]); }

        if(array_key_exists("isDefaultBillingAddress", $data)) {
            $address->setIsDefaultBilling((boolean)$data["isDefaultBillingAddress"]);
        }

        if(array_key_exists("isDefaultShippingAddress", $data)) {
            $address->setIsDefaultShipping((boolean)$data["isDefaultShippingAddress"]);
        }
    }

    public function replaceItem($part) {
        // PUT
        $id = $this->getSegment(3);

        switch($part) {
        case "addresses":
            return $id ? $this->updateCustomerAddress($id) : [405];
        default:
            return [404];
        }
    }

    public function deleteItem($part) {
        // DELETE
        $id = $this->getSegment(3);

        switch($part) {
        case "addresses":
            return $id ? $this->deleteCustomerAddress($id) : [405];
        default:
            return [404];
        }
    }

    public function addToItem($part) {
        // POST
        $id = $this->getSegment(3);

        switch($part) {
        case "login":
            return $id ? [404] : $this->doLogin();
        case "addresses":
            return $id ? [404] : $this->addCustomerAddress();
        case "forgotpassword":
            return $id ? [404] : $this->forgotPassword();
        case "confirm":
            return $id ? [404] : $this->confirmCustomer();
        case "resend_confirmation":
            return $id ? [404] : $this->resendConfirmationEmail();
        default:
            return [404];
        }
    }

    /**
     * @api {post} /customer/confirm Confirm a user
     * @apiName confirmCustomerEmail
     * @apiDescription Successful confirmaton will log in the user.
     *                 Note that the session cookie might be regenerated.
     * @apiGroup Customer
     *
     * @apiParam {integer} id
     * @apiParam {string}  token
     * @apiUse customerResponse
     */
    public function confirmCustomer() {
        $data   = $this->requestData();
        $helper = Mage::helper("API/customer");

        if( ! array_key_exists("id", $data)) {
            throw Crossroads_API_ResponseException::create(400, "Missing id parameter in request body.", null, 3014);
        }

        if( ! array_key_exists("token", $data)) {
            throw Crossroads_API_ResponseException::create(400, "Missing token parameter in request body.", null, 3015);
        }

        $session  = Mage::getSingleton("customer/session");

        if($session->isLoggedIn()) {
            $session->logout()->regenerateSessionId();
        }

        $customer = Mage::getModel("customer/customer")->load($data["id"]);

        if( ! $customer->getId()) {
            throw Crossroads_API_ResponseException::create(404, "User not found", null, 3016);
        }

        if( ! $customer->getConfirmation()) {
            throw Crossroads_API_ResponseException::create(403, "User already active", null, 3017);
        }

        if($customer->getConfirmation() !== $data["token"]) {
            throw Crossroads_API_ResponseException::create(403, "Invalid token", null, 3018);
        }

        $customer->setConfirmation(null)->save();

        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        return [200, $helper->prepareCustomer($customer)];
    }

    /**
     * @api {post} /customer/resend_confirmation Ask to resend the confirmation email
     * @apiName resendConfirmationEmail
     * @apiGroup Customer
     *
     * @apiParam {string}  email
     */
    public function resendConfirmationEmail() {
        $data = $this->requestData();
        $store = Mage::app()->getStore();

        if( ! is_array($data) || ! array_key_exists("email", $data) || ! $data["email"]) {
            throw Crossroads_API_ResponseException::create(400, "Missing email parameter in request body.", null, 3019);
        }

        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail((string)$data["email"]);

        if( ! $customer || ! $customer->getId()) {
            throw Crossroads_API_ResponseException::create(404, "Email not found.", null, 3020);
        }

        if( ! $customer->getConfirmation()) {
            throw Crossroads_API_ResponseException::create(400, "Email does not need to be verified.", null, 3021);
        }

        $customer->sendNewAccountEmail("confirmation", "", $store->getId());

        return [204];
    }

    /**
     * @apiDefine addressRequest
     *
     * @apiParam {string}   [prefix]
     * @apiParam {string}   firstname
     * @apiParam {string}   [middlename]
     * @apiParam {string}   lastname
     * @apiParam {string}   [suffix]
     * @apiParam {string}   [company]
     * @apiParam {string[]} street
     * @apiParam {string}   postcode
     * @apiParam {string}   city
     * @apiParam {string}   [regionId]
     * @apiParam {string}   countryId (SE, NO, DK)
     * @apiParam {string}   [telephone]
     * @apiParam {string}   [fax]
     * @apiParam {boolean}  [isDefaultBillingAddress=false]
     * @apiParam {boolean}  [isDefaultShippingAddress=false]
     */

    /**
     * @apiDefine addressResponse
     *
     * @apiSuccess {integer}  id
     * @apiSuccess {string}   [prefix]
     * @apiSuccess {string}   firstname
     * @apiSuccess {string}   [middlename]
     * @apiSuccess {string}   lastname
     * @apiSuccess {string}   [suffix]
     * @apiSuccess {string}   [company]
     * @apiSuccess {string[]} street
     * @apiSuccess {string}   postcode
     * @apiSuccess {string}   city
     * @apiSuccess {string}   [regionId]
     * @apiSuccess {string}   countryId (SE, NO, DK)
     * @apiSuccess {string}   [telephone]
     * @apiSuccess {string}   [fax]
     * @apiSuccess {boolean}  isDefaultBillingAddress
     * @apiSuccess {boolean}  isDefaultShippingAddress
     */

    /**
     * @apiDefine customerRequest
     *
     * @apiParam {string} email
     * @apiParam {string} [prefix]
     * @apiParam {string} [firstname]
     * @apiParam {string} [middlename]
     * @apiParam {string} [lastname]
     * @apiParam {string} [suffix]
     * @apiParam {string} [VATno]
     * @apiParam {string} [dob]  ISO8601 Date
     * @apiParam {string} [defaultPaymentMethod] Defualt payment method code
     * @apiParam {object} [attributes] Key-value attribute storage, configurable in admin
     */

    /**
     * @apiDefine customerResponse
     *
     * @apiSuccess {integer} id
     * @apiSuccess {string} email
     * @apiSuccess {string} [prefix]
     * @apiSuccess {string} [firstname]
     * @apiSuccess {string} [middlename]
     * @apiSuccess {string} [lastname]
     * @apiSuccess {string} [suffix]
     * @apiSuccess {string} [VATno]
     * @apiSuccess {string} [dob]  ISO8601 Date
     * @apiSuccess {object} [group]
     * @apiSuccess {integer} group.id
     * @apiSuccess {string} group.code
     * @apiSuccess {string} [defaultPaymentMethod] Defualt payment method code
     * @apiSuccess {string} [defaultShippingMethod] Defualt shipping method code
     * @apiSuccess {object} attributes Key-value attribute storage, configurable in admin
     * @apiSuccess {boolean} isConfirmationRequired  If true, the customer needs to confirm the email address before being able to log in
     * @apiSuccess {string} createdAt  ISO8601 Datetime in UTC
     * @apiSuccess {string} updatedAt  ISO8601 Datetime in UTC
     * @apiSuccessExample {json} Example
     * { "id": "38", "email": "test@example.com", "prefix": "Mr", "firstname": "Test", "middlename": "T.", "lastname": "Testsson", "suffix": "Tester", "VATno": "VAT1234", "dob": "2017-03-16", "createdAt": "2017-03-02T12:40:01Z", "updatedAt": "2017-03-02T13:13:46Z" }
     */
    public function getItem($part) {
        $id = $this->getSegment(3);

        switch($part) {
        case "addresses":
            return $id ? $this->getCustomerAddress($id) : $this->getAllCustomerAddresses();
        case "email_available":
            return $this->checkEmailAvailable();
        case "availablePaymentMethods":
            return $this->getAvailablePaymentmethods();
        case "availableShippingMethods":
            return $this->getAvailableShippingmethods();
        default:
            return [404];
        }
    }

    /**
     * @api {get} /customer/addresses List addresses
     * @apiName getAllCustomerAddresses
     * @apiGroup Customer
     * @apiUse addressResponse
     */
    protected function getAllCustomerAddresses() {
        $helper     = Mage::helper("API/customer");
        $addrHelper = Mage::helper("API/address");

        $this->ensureLoggedIn();

        $customer  = $helper->getCurrentCustomer();
        $addresses = new ArrayObject(array_map(function($a) use($addrHelper, $customer) {
            return $addrHelper->prepareCustomerAddress($a, $customer);
        }, $customer->getAddresses()));

        Mage::dispatchEvent(self::EVENT_CUSTOMER_ADDRESSES_LIST_POST_DATA_PREPARE, [
            "customer" => $customer,
            "addresses" => $addresses,
        ]);

        return [200, array_values($addresses->getArrayCopy())];
    }

    /**
     * @api {get} /customer/addresses/:id Get address
     * @apiName getCustomerAddress
     * @apiGroup Customer
     * @apiUse addressResponse
     */
    protected function getCustomerAddress($id) {
        $helper     = Mage::helper("API/customer");
        $addrHelper = Mage::helper("API/address");

        $this->ensureLoggedIn();

        $customer = $helper->getCurrentCustomer();
        $address  = $customer->getAddressById($id);

        return $address && $address->getId() == $id ? [200, $addrHelper->prepareCustomerAddress($address, $customer)] : [404];
    }

    /**
     * @api {get} /customer/email_available  Check email availability
     * @apiName checkCustomerEmailAvailable
     * @apiGroup Customer
     * @apiDescription Does not require the user to be logged in.
     *
     * @apiParam   {String}  email   The email address to check
     * @apiSuccess {boolean} available
     */
    protected function checkEmailAvailable() {
        $helper  = Mage::helper("API/customer");
        $website = Mage::app()->getWebsite();
        $email   = trim($this->getRequest()->getQuery("email"));

        if(!$email) {
            throw Crossroads_API_ResponseException::create(400, "Missing 'email' parameter.", null, 3012);
        }

        return [200, [
            "available" => $helper->isEmailAvailable($email, $website->getId()),
        ]];
    }

    /**
     * @api {get} /customer/availablePaymentMethods  List available payment methods
     * @apiName  getAvailablePaymentmethods
     * @apiGroup Customer
     * @apiDescription Lists all payment methods which can be selected as the
     *                 default payment method
     *
     * @apiSuccess  {String}  code
     * @apiSuccess  {String}  label
     */
    protected function getAvailablePaymentmethods() {
        return [200, array_values(array_map(function($p) {
            return [
                "code"  => $p->getCode(),
                "label" => $p->getTitle(),
            ];
        }, Mage::helper("API/payment")->getEnabledMethods()))];
    }

    /**
     * @api {get} /customer/availableShippingMethods  List available shipping methods
     * @apiName  getAvailableShippingmethods
     * @apiGroup Customer
     * @apiDescription Lists all shipping methods which can be selected as the
     *                 default payment method
     *
     * @apiSuccess  {String}  code
     * @apiSuccess  {String}  title
     * @apiSuccess  {String}  label
     * @apiSuccess  {String}  description
     */
    protected function getAvailableShippingmethods() {
        $this->ensureLoggedIn();

        return [200, array_values(array_map(function($p) {
            return [
                "code"        => sprintf("%s_%s", $p->getCarrier(), $p->getMethod()),
                "title"       => $p->getCarrierTitle(),
                "label"       => $p->getMethodTitle(),
                "description" => $p->getMethodDescription(),
            ];
        }, Mage::helper("API/shipping")->getAllShippingRatesForCustomer(Mage::getSingleton("customer/session")->getCustomer())))];
    }

    /**
     * @api {post} /customer/addresses Create address
     * @apiName addCustomerAddress
     * @apiGroup Customer
     * @apiUse addressRequest
     * @apiUse addressResponse
     */
    protected function addCustomerAddress() {
        $this->ensureLoggedIn();

        $data       = $this->requestData();
        $helper     = Mage::helper("API/customer");
        $addrHelper = Mage::helper("API/address");
        $customer   = $helper->getCurrentCustomer();
        $address    = Mage::getModel("customer/address");

        if( ! $data || ! is_array($data)) {
            throw Crossroads_API_ResponseException::create(400, "Address data must be an object", null, 3008);
        }

        $this->loadAddressData($address, $data);

        $address->setCustomerId($customer->getId());
        $address->setSaveInAddressBook("1");

        $errors = $address->validate();

        if($errors !== true) {
            throw Crossroads_API_ResponseException::create(400, "Address failed to validate", $errors, 3007);
        }

        $customer->addAddress($address);

        $address->save();

        $this->updateDefaultAddresses($customer, $address);

        $customer->save();

        return [200, $addrHelper->prepareCustomerAddress($address, $customer)];
    }

    /**
     * @api {put} /customer/addresses/:id Update address
     * @apiName updateCustomerAddress
     * @apiGroup Customer
     * @apiUse addressRequest
     * @apiUse addressResponse
     */
    protected function updateCustomerAddress($id) {
        $this->ensureLoggedIn();

        $data       = $this->requestData();
        $helper     = Mage::helper("API/customer");
        $addrHelper = Mage::helper("API/address");
        $customer   = $helper->getCurrentCustomer();
        $address    = $customer->getAddressById($id);

        if( ! $address || $address->getId() != $id) {
            return [404];
        }

        if( ! $data || ! is_array($data)) {
            throw Crossroads_API_ResponseException::create(400, "Address data must be an object", null, 3008);
        }

        $this->loadAddressData($address, $data);

        $address->setCustomerId($customer->getId());
        $address->setSaveInAddressBook("1");

        $errors = $address->validate();

        if($errors !== true) {
            throw Crossroads_API_ResponseException::create(400, "Address failed to validate", $errors, 3007);
        }

        $address->save();

        $this->updateDefaultAddresses($customer, $address);

        $customer->save();

        return [200, $addrHelper->prepareCustomerAddress($address, $customer)];
    }

    /**
     * Updates customer and quote addresses if the address is a default shipping/billing
     * address.
     *
     * @param  Mage_Customer_Model_Customer
     * @param  Mage_Customer_Model_Address_Abstract
     */
    protected function updateDefaultAddresses($customer, $address) {
        $quote    = Mage::getSingleton("checkout/session")->getQuote();
        $modified = false;

        if($address->getIsDefaultBilling()) {
            $customer->setDefaultBilling($address->getId());

            if($quote &&
               ($billingAddr = $quote->getBillingAddress()) &&
               $billingAddr->getId() &&
               $billingAddr->getCustomerAddressId()) {
                $billingAddr->importCustomerAddress($address);

                $modified = true;
            }
        }

        if($address->getIsDefaultShipping()) {
            $customer->setDefaultShipping($address->getId());

            if($quote &&
               ($shippingAddr = $quote->getShippingAddress()) &&
               $shippingAddr->getId() &&
               $shippingAddr->getCustomerAddressId()) {
                $shippingAddr->importCustomerAddress($address)
                    ->setCollectShippingRates(true);

                $modified = true;
            }
        }

        if($modified) {
            $quote->save();
        }
    }

    /**
     * @api {delete} /customer/addresses/:id Delete address
     * @apiName deleteCustomerAddress
     * @apiGroup Customer
     */
    protected function deleteCustomerAddress($id) {
        $helper     = Mage::helper("API/customer");
        $addrHelper = Mage::helper("API/address");

        $this->ensureLoggedIn();

        $customer = $helper->getCurrentCustomer();
        $address  = $customer->getAddressById($id);

        if( ! $address || $address->getId() != $id) {
            return [404];
        }

        $address->delete();

        return [204];
    }

    /**
     * @api {get} /customer Get logged in customer
     * @apiName getLoggedInCustomer
     * @apiGroup Customer
     * @apiUse customerResponse
     */
    public function getAll() {
        $this->ensureLoggedIn();

        $helper   = Mage::helper("API/customer");
        $customer = $helper->getCurrentCustomer();

        return [200, $helper->prepareCustomer($customer)];
    }

    /**
     * @api {post} /customer Register
     * @apiName registerCustomer
     * @apiGroup Customer
     * @apiDescription If a user is successfully created and Magento is configured to not require
     * an email confirmation, the customer will automatically be logged in. The application can
     * check for this by looking at the `isConfirmationRequired` property on the returned
     * customer object, if it is `false` the customer is already logged in.
     * @apiUse customerRequest
     * @apiParam {string} [defaultShippingMethod]
     * @apiParam {string} [password]
     * @apiUse customerResponse
     */
    public function createItem() {
        $helper = Mage::helper("API/customer");
        $session  = Mage::getSingleton("customer/session");

        if( ! $helper->allowRegister()) {
            throw Crossroads_API_ResponseException::create(403, "Customer registration is disabled.", null, 3001);
        }

        $data = $this->requestData();

        if( ! $data || ! is_array($data)) {
            throw Crossroads_API_ResponseException::create(400, "Customer data must be an object", null, 3003);
        }

        try {
            $customer = $helper->createCustomer($data);
        }
        catch(Mage_Eav_Model_Entity_Attribute_Exception $e) {
            throw Crossroads_API_ResponseException::create(400, "Customer data failed validation", $e->getMessage(), 3004);
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper('customer')->__('This customer email already exists')) {
                throw Crossroads_API_ResponseException::create(403, "Customer email address is already in use", null, 3005);
            }

            throw $e;
        }

        if($customer->isConfirmationRequired()) {
            $customer->sendNewAccountEmail(
                'confirmation',
                $session->getBeforeAuthUrl(),
                Mage::app()->getStore()->getId(),
                $data["password"]
            );
        }
        else {
            $session->setCustomerAsLoggedIn($customer);

            $customer->sendNewAccountEmail(
                'registered', // confirmed on email confirmation
                '',
                Mage::app()->getStore()->getId(),
                $this->getRequest()->getPost('password')
            );
        }

        return [201, $helper->prepareCustomer($customer)];
    }

    /**
     * @api {put} /customer Update customer data
     * @apiName updateCustomer
     * @apiGroup Customer
     * @apiUse customerRequest
     * @apiParam {string} [password]
     * @apiUse customerResponse
     */
    public function replaceAll() {
        $this->ensureLoggedIn();

        $helper = Mage::helper("API/customer");
        $data   = $this->requestData();

        if( ! $data || ! is_array($data)) {
            throw Crossroads_API_ResponseException::create(400, "Customer data must be an object", null, 3003);
        }

        try {
            $customer = $helper->updateCustomer($data);
        }
        catch(Mage_Eav_Model_Entity_Attribute_Exception $e) {
            throw Crossroads_API_ResponseException::create(400, "Customer data failed validation", $e->getMessage(), 3004);
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper('customer')->__('This customer email already exists')) {
                throw Crossroads_API_ResponseException::create(403, "New customer email address is already in use", null, 3009);
            }

            throw $e;
        }

        return [200, $helper->prepareCustomer($customer)];
    }

    /**
     * @api {delete} /customer  Logout
     * @apiName logoutCustomer
     * @apiGroup Customer
     */
    public function deleteAll() {
        try {
            Mage::getSingleton("customer/session")->logout();
        }
        catch(Mage_Core_Model_Session_Exception $e) {
            // intentionally left empty
        }

        return [204];
    }

    /**
     * @api {post} /customer/login Login
     * @apiName loginCustomer
     * @apiGroup Customer
     * @apiParam {string} email
     * @apiParam {string} password
     * @apiUse customerResponse
     */
    protected function doLogin() {
        $helper = Mage::helper("API/customer");
        $data   = $this->requestData();

        if( ! $data || ! is_array($data)) {
            throw Crossroads_API_ResponseException::create(400, "Customer data must be an object", null, 3003);
        }

        if(empty($data["email"])) {
            throw Crossroads_API_ResponseException::create(400, "Customer is missing email", null, 3002);
        }

        if(empty($data["password"])) {
            throw Crossroads_API_ResponseException::create(400, "Customer is missing password", null, 3002);
        }

        try {
            $customer = $helper->loginCustomer($data["email"], $data["password"]);
        }
        catch(Mage_Core_Exception $e) {
            if($e->getMessage() === Mage::helper('customer')->__('Invalid login or password.')) {
                throw Crossroads_API_ResponseException::create(403, "Invalid email or password.", null, 3006);
            }

            if($e->getMessage() === Mage::helper("customer")->__("This account is not confirmed.")) {
                throw Crossroads_API_ResponseException::create(403, "The account is not confirmed by email.", null, 3013);
            }

            throw $e;
        }

        return [200, $helper->prepareCustomer($customer)];
    }

    /**
     * @api {post} /customer/forgotpassword Forgot password
     * @apiName forgotPassword
     * @apiGroup Customer
     * @apiParam {string} email
     * @apiUse customerResponse
     */
    protected function forgotPassword() {
        $helper = Mage::helper("API/customer");
        $data   = $this->requestData();

        if (empty($data["email"]) || strlen(trim($data["email"])) === 0) {
            throw Crossroads_API_ResponseException::create(400, "Email field is missing", null, 3010);
        }

        $customer = Mage::getModel('customer/customer')
            ->setWebsiteId(Mage::app()->getStore()->getWebsiteId())
            ->loadByEmail($data["email"]);

        if (!$customer->getId()) {
            throw Crossroads_API_ResponseException::create(400, "Customer not found", null, 3011);
        }

        $newResetPasswordLinkToken = Mage::helper('customer')->generateResetPasswordLinkToken();
        $customer->changeResetPasswordLinkToken($newResetPasswordLinkToken);
        $customer->sendPasswordResetConfirmationEmail();

        return [204];
    }
}
