<?php

class Crossroads_API_RouteController extends Crossroads_API_Controller_Super
{
    const BASE_URL = "/api/route";

    public function indexAction() {
        return $this->sendData([404]);
    }

    // Using true here we tell Magento's router that we support all actions.
    public function hasAction($_action) { return true; }

    /**
     * @api {get} /route/:url Get dynamic resource by url
     * @apiName route
     * @apiGroup Route
     * @apiDescription Uses Magento's `url_rewrite` module to determine if a category
     *     or product resides at that URL. If so that category/product/redirect/CMS-page will be returned.
     *
     * @apiParam {String} url The slug to search the index by
     * @apiParam {Integer} [limit=20] The pagination size for category contents, max 100
     * @apiParam {Integer} [page=1] The pagination page for category contents
     *
     * @apiSuccess {String} type Type of resource
     * @apiSuccess {Object} product Product object
     * @apiSuccess {Object} cmspage CMS page object
     * @apiSuccess {Object} category Category object
     * @apiSuccess {Object} redirect Redirect URL
     * @apiSuccess {String} redirect.url Actual redirect URL, if it does not start with a protocol it is relative to this endpoint
     * @apiSuccess {Boolean} redirect.permanent If the redirect is permanent
     */
    public function norouteAction($_core_route = null)
    {
        if($this->getRequest()->getMethod() !== "GET") {
            return $this->sendData([405]);
        }

        $url = $this->getRequest()->getRequestUri();

        if(strpos($url, self::BASE_URL) !== 0) {
            return $this->sendData([404]);
        }

        // Cut off the start of the URL and everything from ? to the end, if ? is present in the string
        // Note the use of before-needle in strstr
        $url = substr($url, strlen(self::BASE_URL));
        $url = trim(strstr($url, "?", true) ?: $url, "/");

        $rewrite = Mage::getModel("core/url_rewrite")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->loadByRequestPath($url);

        $isRedirect  = stripos($rewrite->getOptions(), "R") !== false;
        $isPermanent = stripos($rewrite->getOptions(), "P") !== false;

        // is redirect
        if ($isRedirect && $rewrite->getRequestPath() && $rewrite->getTargetPath() &&
            !$rewrite->getCategoryId() && !$rewrite->getProductId()) {

            return $this->sendData([200, [
                "type"     => "redirect",
                "product"  => null,
                "cmsPage"  => null,
                "category" => null,
                "redirect" => [
                    "url"       => $rewrite->getTargetPath(),
                    "permanent" => $isPermanent
                ]
            ]]);
        }

        // is product
        if ($rewrite->getProductId()) {
            $product = Mage::getModel('catalog/product')->load($rewrite->getProductId());

            if( ! $product || ! $product->getId() || ! $product->isVisibleInSiteVisibility()) {
                return $this->sendData([404]);
            }

            return $this->sendData([200, [
                "type"     => "product",
                "product"  => Mage::helper("API/product")->prepareProductDetail($product),
                "cmsPage"  => null,
                "category" => null,
                "redirect" => null
            ]]);
        }

        // is category
        if ($rewrite->getCategoryId()) {
            $category = Mage::getModel("catalog/category")->load($rewrite->getCategoryId());
            $pageSize = min((int)$this->getRequest()->getQuery("limit", "20"), 100);
            $page     = max((int)$this->getRequest()->getQuery("page", "1"), 1);

            if (!$category || !$category->getId() || !$category->getIsActive()) {
                return $this->sendData([404]);
            }

            $catHelper  = Mage::helper("API/category");
            $collection = $catHelper->createProductCollection($category);
            // We call this directly on the select to prevent magento from stopping us from
            // going off the end of the list. (setCurPage() prevents this, which is not correct
            // from an API PoV)
            $collection->getSelect()->limitPage($page, $pageSize);
            $collection->load();
            $pHelper    = Mage::helper("API/product");

            return $this->sendData([200, [
                "type"     => "category",
                "product"  => null,
                "cmsPage"  => null,
                "category" => [
                    "category" => $catHelper->prepareCategory($category, true, $collection),
                    "products" => array_values(array_map(function($p) use($pHelper) {
                        return $pHelper->prepareListProduct($p);
                    }, $collection->getItems()))
                ],
                "redirect" => null
            ]]);
        }

        // is cms page
        $id = Mage::helper('API/cms')->getCategoryIdByUrlKey($url);

        if (!empty($id)) {
            $page = Mage::helper('API/cms')->getItem($id);

            if ($page) {
                return $this->sendData([200, [
                    "type"     => "cmspage",
                    "product"  => null,
                    "cmsPage"  => $page,
                    "category" => null
                ]]);
            }
        }

        return $this->sendData([404]);
    }
}
