<?php

class Crossroads_API_Controller_Super extends Mage_Core_Controller_Front_Action
{
    /**
     * Dispatch event before action, overridden to prevent 302 Found on failure to
     * start session.
     *
     * @return void
     */
    public function preDispatch() {
        // From Mage_Core_Controller_Front_Action::preDispatch()
        $this->getLayout()->setArea($this->_currentArea);

        // Adapted from Mage_Core_Controller_Varien_Action::preDispatch()
        if( ! Mage::isInstalled()) {
            // Prevent dispatch
            $this->setFlag("", self::FLAG_NO_DISPATCH, true);
            $this->getResponse()
                ->setHttpResponseCode(500)
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([
                    "message"   => "Magento is not installed properly.",
                    "errorCode" => 1100,
                    "data"      => null,
                ]));

            return;
        }

        if( ! Mage::app()->getStore()->getIsActive()) {
            // TODO: can we do something better here? this results in the include of
            // errors/404.php
            Mage::app()->throwStoreException();
        }

        // Boot session, attempt to restart it if it is invalid.
        //
        // Alternate version of the session-start code in
        // Mage_Core_Controller_Varien_Action::preDispatch(), which had the issue
        // of failing to restart the session properly and instead causing errors
        // to become redirects which is incompatible with an API.
        try {
            Mage::getSingleton("core/session", ["name" => $this->_sessionNamespace])
                ->start();
        }
        catch(Mage_Core_Model_Session_Exception $_e) {
            // The session is invalid, destroy if it is active
            if(session_status() == PHP_SESSION_ACTIVE) {
                session_destroy();
                session_commit();
            }

            // New id to reinitialize
            session_regenerate_id(true);

            // Try to reinitialize session
            try {
                Mage::getSingleton("core/session", ["name" => $this->_sessionNamespace])
                    ->start()
                    // Core is from Mage_Core_Model_Session::__construct()
                    // init is called to make sure the session is properly initialized
                    ->init("core", $this->_sessionNamespace);
            }
            catch(Exception $e) {
                Mage::logException($e);

                $this->setFlag("", self::FLAG_NO_DISPATCH, true);
                $this->getResponse()
                    ->setHttpResponse(400)
                    ->setHeader("Content-Type", "application/json", true)
                    ->setBody(json_encode([
                        "message"   => "Failed to reinitialize session",
                        "errorCode" => 1101,
                        "data"      => null,
                    ]));

                return;
            }
        }
        catch(Exception $e) {
            $this->setFlag("", self::FLAG_NO_DISPATCH, true);

            return $this->handleException($e);
        }

        // Code from Mage_Core_Controller_Varien_Action::preDispatch() again
        Mage::app()->loadArea($this->getLayout()->getArea());

        Varien_Autoload::registerScope($this->getRequest()->getRouteName());

        Mage::dispatchEvent("controller_action_predispatch", ["controller_action" => $this]);
        Mage::dispatchEvent("controller_action_predispatch_" . $this->getRequest()->getRouteName(),
            ["controller_action" => $this]);
        Mage::dispatchEvent("controller_action_predispatch_" . $this->getFullActionName(),
            ["controller_action" => $this]);
    }

    /**
     * For outputting a JSON response
     *
     * @param int $status, HTTP code
     * @param mixed $data, the response data
     */
    protected function sendData($data) {
        if(!is_array($data) || !$data[0]) {
            throw new Exception("Missing return array.");
        }

        assert(gettype($data[0]) === "integer", __METHOD__.": \$status must be of type integer");

        $this->getResponse()->setHttpResponseCode($data[0]);

        if(array_key_exists(1, $data)) {
            $this->getResponse()
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode($data[1]));
        }
    }

    /**
     * Get JSON request data as an assoc array, will output null if Content-Type
     * is not application/json.
     *
     * NOTE: Does not accept NULL as a json string.
     *
     * @return array|null attached data, null if not application/json
     * @throws Crossroads_API_ResponseException  If the content is application/json but cannot be parsed
     */
    protected function requestData() {
        $request      = $this->getRequest();
        $content_type = $request->getHeader("Content-Type");

        if(strtolower(trim($content_type)) === "application/json") {
            $data = json_decode($request->getRawBody(), true);

            if($data === null) {
                throw Crossroads_API_ResponseException::create(400, "Could not parse JSON data", null, 1001);
            }

            return $data;
        }

        return null;
    }

    /**
     * Utility function ensuring that the user is logged in, throwing a
     * response-exception if the user is not logged in.
     *
     * @throws Crossroads_API_ResponseException
     */
    protected function ensureLoggedIn() {
        if( ! Mage::helper("API/customer")->isLoggedIn()) {
            throw Crossroads_API_ResponseException::create(403, "Customer is not logged in", null, 3000);
        }
    }

    /**
     * This method attempts to reinitialize the checkout session if it is no longer valid.
     *
     * @return Mage_Checkout_Model_Session
     */
    protected function getCheckoutSession() {
        try {
            $sess = Mage::getSingleton("checkout/session");

            $this->loadDefaultCustomerData($sess->getQuote());

            return $sess;
        }
        catch(Mage_Core_Model_Session_Exception $e) {
            Mage::log(sprintf("%s: %s, Got a %s, resetting checkout session.",
                __METHOD__,
                Mage::helper("core/http")->getRemoteAddr(true),
                get_class($e)));

            unset($_SESSION["checkout"]);

            $sess = Mage::getSingleton("checkout/session");

            $this->loadDefaultCustomerData($sess->getQuote());

            return $sess;
        }
    }

    /**
     * Returns the contents of the n:th URL segment.
     *
     * @param  integer
     * @return string
     */
    protected function getSegment($num) {
        return current(array_slice(explode("/", trim($this->getRequest()->getRequestUri(), "/")), $num, 1));
    }

    /**
     * This method attempts to reinitialize the checkout session if it is no longer valid.
     *
     * @return Mage_Checkout_Model_Cart
     */
    protected function getCheckoutCart() {
        try {
            $cart = Mage::getSingleton("checkout/cart");

            $this->loadDefaultCustomerData($cart->getQuote());

            return $cart;
        }
        catch(Mage_Core_Model_Session_Exception $e) {
            Mage::log(sprintf("%s: %s, Got a %s, resetting checkout session.",
                __METHOD__,
                Mage::helper("core/http")->getRemoteAddr(true),
                get_class($e)));

            unset($_SESSION["checkout"]);

            $cart = Mage::getSingleton("checkout/cart");

            $this->loadDefaultCustomerData($cart->getQuote());

            return $cart;
        }
    }

    /**
     * Attempts to load default customer data into a quote.
     *
     * Customer will be assigned if the quote does not already have that
     * particular customer assigned. Addresses will be assigned if they are
     * empty according to Crossroads_API_Helper_Address::addressIsEmpty().
     *
     * @param  Mage_Sales_Model_Quote
     * @return void
     */
    protected function loadDefaultCustomerData($quote) {
        $sess     = Mage::getSingleton("customer/session");
        $helper   = Mage::helper("API/address");
        $customer = $sess->getCustomer();
        $modified = false;

        if($sess->isLoggedIn() && $customer->getId()) {
            if($quote->getCustomerId() != $customer->getId()) {
                $quote->setCustomer($customer);

                $modified = true;
            }

            if($helper->addressIsEmpty($quote->getBillingAddress()) &&
               ($defaultBillingAddress = $customer->getDefaultBillingAddress()) &&
               $defaultBillingAddress->getId()) {
                $quote->setBillingAddress(Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultBillingAddress));

                $modified = true;
            }

            if($helper->addressIsEmpty($quote->getShippingAddress()) &&
               ($defaultShippingAddress = $customer->getDefaultShippingAddress()) &&
               $defaultShippingAddress->getId()) {
                $quote->setShippingAddress(Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultShippingAddress))
                    ->setTotalsCollectedFlag(false)
                    ->getShippingAddress()
                    ->setCollectShippingRates(true);

                $modified = true;
            }

            if($modified) {
                $quote->save();
            }
        }
    }

    /**
     * Helper function to handle an exception which is at an unrecoverable
     * (or outermost) point in the controller, it will ensure that
     * response-exceptions are not logged and become responses while actual
     * exceptions are logged, and if developer mode is on they are also emitted
     * as JSON in the response body.
     *
     * @param  Exception
     * @return void
     */
    protected function handleException($e) {
        if($e instanceof Crossroads_API_ResponseException) {
            // Crossroads_API_ResponseExceptions should not be logged, they are responses
            // to be sent to the client
            return $this->getResponse()
                ->setHttpResponseCode($e->getStatusCode())
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([
                "message"   => $e->getMessage(),
                "errorCode" => $e->getCode(),
                "data"      => $e->getData()
            ]));
        }

        $this->getResponse()->setHttpResponseCode(500);

        Mage::logException($e);

        if(Mage::getIsDeveloperMode()) {
            $this->getResponse()
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([
                    "message"   => $e->getMessage(),
                    "errorCode" => null,
                    "data"      => [
                        "code"  => $e->getCode(),
                        "file"  => $e->getFile(),
                        "line"  => $e->getLine(),
                        "trace" => array_map(function($l) {
                            if(array_key_exists("class", $l)) {
                                return [
                                    "call" => $l["class"].$l["type"].$l["function"],
                                    "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                    "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                ];
                            }

                            return [
                                "call" => $l["function"],
                                "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                "line" => array_key_exists("line", $l) ? $l["line"] : null,
                            ];
                        }, $e->getTrace())
                    ]
                ]));
        }
    }
}
