<?php

/**
 * Helper managing payment methods through an instance of Crossroads_API_Payment_Registry.
 * Payment methods are registered through the "crossroads_api_payment_reigster" event.
 */
class Crossroads_API_Helper_Payment {
    /**
     * The payment base URL which is to be stripped before considering routing.
     *
     * This is placed here since autoloading controllers is not supported.
     */
    const BASE_URI = "/api/payment";

    /**
     * Event triggered when payment-methods are initialized. This event is used to register
     * payment methods with Crossroads_API.
     *
     * Params:
     *  * registry  The Crossroads_API_Payment_Registry instance to reigster methods with
     */
    const EVENT_REGISTER_PAYMENT_METHODS = "crossroads_api_payment_register";

    /**
     * @var Crossroads_API_Payment_Registry
     */
    protected $registry;

    /*
     * Populates a payment registry with payment methods registered through the
     * "crossroads_api_payment_register" event.
     */
    public function __construct() {
        $this->registry = new Crossroads_API_Payment_Registry();

        Mage::dispatchEvent(self::EVENT_REGISTER_PAYMENT_METHODS, [
            "registry" => $this->registry
        ]);
    }

    /**
     * Returns true if a payment method with the supplied payment code exists in the registry.
     *
     * @param  string
     * @return boolean
     */
    public function hasMethod($code) {
        return $this->registry->hasMethod($code);
    }

    /**
     * Returns the payment model associated with the supplied payment code, throws an exception
     * if the associated payment method does not exist.
     *
     * @param  string
     * @return Crossroads_API_Payment_Abstract
     */
    public function getMethod($code) {
        return $this->registry->getMethod($code);
    }

    /**
     * Returns a list of supported AND enabled payment methods.
     *
     * @param  Mage_Sales_Model_Quote
     * @param  Mage_Core_Model_Store
     * @return Array<Crossroads_API_Payment_Abstract>
     */
    public function getEnabledMethods($quote = null, $store = null) {
        $methods = Mage::getSingleton('payment/config')->getActiveMethods($store);
        $codes   = array_keys(array_filter($methods, function($method) use($quote, $store) {
            if($store) {
                $method->setStore($store);
            }

            return $method->isAvailable($quote);
        }));
        $methods = [];

        foreach($codes as $code) {
            if($this->registry->hasMethod($code)) {
                $methods[] = $this->registry->getMethod($code);
            }
        }

        return $methods;
    }

    public function isValidPaymentMethod($code, $quote = null, $store = null) {
        foreach($this->getEnabledMethods($quote, $store) as $m) {
            if($m->getCode() === $code) {
                return true;
            }
        }

        return false;
    }
}
