<?php

class Crossroads_API_Helper_Attributes {
    /**
     * Executed after an associated array has been prepared for the attribute-configuration,
     * this array can be modified.
     *
     * Params:
     *  * entity:        Entity
     *  * config:        Configuration array of frontend_input and attribute_code
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_POST_DATA_PREPARE = "crossroads_api_attributes_post_data_prepare";

    /**
     * Retrieves the entity attribute text for the given entity, if it is using EAV.
     * Falls back to the value of `getData($attributeCode)`.
     *
     * @param  Mage_Core_Model_Abstract
     * @param  string
     * @param  boolean
     * @return string|null
     */
    public function getEntityAttributeText($entity, $attributeCode, $multiselect = false) {
        $data = $entity->getData($attributeCode);

        // Essentially copied from Mage_Catalog_Model_Product::getAttributeText
        // This is missing on Categories and Customer, and here modified to also support
        // multiselect with array-storage
        $sourceModel = $entity->getResource()->getAttribute($attributeCode)->getSource();

        if( ! $multiselect) {
            return $sourceModel->getOptionText($data) ?: $data;
        }

        return array_filter(array_map([$sourceModel, "getOptionText"], array_map("trim", is_array($data) ? $data : explode(",", $data))));
    }

    public function getAttributeOptionId($entity, $attributeCode, $value) {
        return $entity->getResource()
            ->getAttribute($attributeCode)
            ->getSource()
            ->getOptionId($value);
    }

    /**
     * Fetches the user-configured entity attributes for the supplied entity.
     *
     * @param  Mage_Catalog_Model_Product|Mage_Catalog_Model_Customer
     * @param  array   Array with attribute_code and frontend_input
     * @return object
     */
    public function getEntityAttributes($entity, $config)
    {
        $attributes = [];

        foreach (($config ?: []) as $attribute) {
            if(!is_array($attribute) ||
                !array_key_exists("attribute_code", $attribute) ||
                !array_key_exists("frontend_input", $attribute)) {
                continue;
            }

            $code  = $attribute["attribute_code"];
            $input = $attribute["frontend_input"];

            switch($input) {
                case "select":
                    $attributes[$code] = $this->getEntityAttributeText($entity, $code) ?: null;
                    break;
                case "multiselect":
                    $attributes[$code] = $this->getEntityAttributeText($entity, $code, true) ?: [];
                    break;
                case "media_image":
                    $value       = $this->getEntityAttributeText($entity, $code) ?: null;
                    $mediaConfig = $entity->getMediaConfig();

                    $attributes[$code] = (empty($value) || $value === "no_selection") && $mediaConfig ? null :
                        $mediaConfig->getMediaUrl($value);
                    break;
                default:
                    $attributes[$code] = $entity->getData($code);
            }
        }

        $data = new Varien_Object($attributes);

        Mage::dispatchEvent(self::EVENT_POST_DATA_PREPARE, [
            "entity"        => $entity,
            "config"        => $config,
            "prepared_data" => $data,
        ]);

        return $data->getData() ?: new stdClass();
    }

    /**
     * Updates the custom attributes on the entity based on the configuration.
     *
     * @param  Mage_Customer_Model_Customer
     * @param  array    Array with attribute_code, frontend_input and editable.
     * @param  array    Input data
     * @return object
     */
    public function updateEntityAttributes($entity, $config, $newData) {
        foreach(($config ?: []) as $attribute) {
            if(!is_array($attribute) ||
               !array_key_exists("attribute_code", $attribute) ||
               !array_key_exists("frontend_input", $attribute) ||
               !array_key_exists("editable", $attribute)) {
                continue;
            }

            $code  = $attribute["attribute_code"];
            $input = $attribute["frontend_input"];

            if(strtolower(trim((string)$attribute["editable"])) === "true" && array_key_exists($code, $newData)) {
                $value = $newData[$code];

                switch($input) {
                case "select":
                    if( ! is_scalar($value) && $value !== null) {
                        throw Crossroads_API_ResponseException::create(400, sprintf("Input for entity attribute '%s' expected scalar, got array or object.", $code), null, 1003);
                    }

                    $value = $this->getAttributeOptionId($entity, $code, $value);
                    break;
                case "multiselect":
                    $value = array_unique(array_filter(array_map(function($v) use($entity, $code) {
                        return $this->getAttributeOptionId($entity, $code, $v);
                    }, is_array($value) ? $value : [$value])));
                    break;
                default:
                    // Fallthrough on purpose
                }

                $entity->setData($code, $value);
            }
        }

        return $entity;
    }
}
