<?php

class Crossroads_API_Controller_Super extends Mage_Core_Controller_Front_Action
{
    /**
     * For outputting a JSON response
     *
     * @param int $status, HTTP code
     * @param mixed $data, the response data
     */
    protected function sendData($data) {
        if(!is_array($data) || !$data[0]) {
            throw new Exception("Missing return array.");
        }

        assert(gettype($data[0]) === "integer", __METHOD__.": \$status must be of type integer");

        $this->getResponse()->setHttpResponseCode($data[0]);

        if(array_key_exists(1, $data)) {
            $this->getResponse()
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode($data[1]));
        }
    }

    /**
     * Get JSON request data as an assoc array, will output null if Content-Type
     * is not application/json.
     *
     * NOTE: Does not accept NULL as a json string.
     *
     * @return array|null attached data, null if not application/json
     * @throws Crossroads_API_ResponseException  If the content is application/json but cannot be parsed
     */
    protected function requestData() {
        $request      = $this->getRequest();
        $content_type = $request->getHeader("Content-Type");

        if(strtolower(trim($content_type)) === "application/json") {
            $data = json_decode($request->getRawBody(), true);

            if($data === null) {
                throw Crossroads_API_ResponseException::create(400, "Could not parse JSON data", null, 1001);
            }

            return $data;
        }

        return null;
    }

    protected function ensureLoggedIn() {
        if( ! Mage::helper("API/customer")->isLoggedIn()) {
            throw Crossroads_API_ResponseException::create(403, "Customer is not logged in", null, 3000);
        }
    }

    /**
     * This method attempts to reinitialize the checkout session if it is no longer valid.
     *
     * @return Mage_Checkout_Model_Session
     */
    protected function getCheckoutSession() {
        try {
            $sess = Mage::getSingleton("checkout/session");

            $this->loadDefaultCustomerData($sess->getQuote());

            return $sess;
        }
        catch(Mage_Core_Model_Session_Exception $e) {
            Mage::log(sprintf("%s: %s, Got a %s, resetting checkout session.", __METHOD__, Mage::helper('core/http')->getRemoteAddr(true), get_class($e)));

            unset($_SESSION["checkout"]);

            $sess = Mage::getSingleton("checkout/session");

            $this->loadDefaultCustomerData($sess->getQuote());

            return $sess;
        }
    }

    /**
     * Returns the contents of the n:th URL segment.
     *
     * @param  integer
     * @return string
     */
    protected function getSegment($num) {
        return current(array_slice(explode("/", trim($this->getRequest()->getRequestUri(), "/")), $num, 1));
    }

    /**
     * This method attempts to reinitialize the checkout session if it is no longer valid.
     *
     * @return Mage_Checkout_Model_Cart
     */
    protected function getCheckoutCart() {
        try {
            $cart = Mage::getSingleton("checkout/cart");

            $this->loadDefaultCustomerData($cart->getQuote());

            return $cart;
        }
        catch(Mage_Core_Model_Session_Exception $e) {
            Mage::log(sprintf("%s: %s, Got a %s, resetting checkout session.", __METHOD__, Mage::helper('core/http')->getRemoteAddr(true), get_class($e)));

            unset($_SESSION["checkout"]);

            $cart = Mage::getSingleton("checkout/cart");

            $this->loadDefaultCustomerData($cart->getQuote());

            return $cart;
        }
    }

    protected function loadDefaultCustomerData($quote) {
        $sess     = Mage::getSingleton("customer/session");
        $customer = $sess->getCustomer();
        $modified = false;

        if($sess->isLoggedIn() && $customer->getId()) {
            if($quote->getCustomerId() != $customer->getId()) {
                $quote->setCustomer($customer);

                $modified = true;
            }

            if( ! $quote->getBillingAddress()->getId() &&
               ($defaultBillingAddress = $customer->getDefaultBillingAddress()) &&
               $defaultBillingAddress->getId()) {
                $quote->setBillingAddress(Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultBillingAddress));

                $modified = true;
            }

            if( ! $quote->getShippingAddress()->getId() &&
               ($defaultShippingAddress = $customer->getDefaultShippingAddress()) &&
               $defaultShippingAddress->getId()) {
                $quote->setShippingAddress(Mage::getModel("sales/quote_address")
                    ->importCustomerAddress($defaultShippingAddress))
                    ->setTotalsCollectedFlag(false)
                    ->getShippingAddress()
                    ->setCollectShippingRates(true);

                $modified = true;
            }

            if($modified) {
                $quote->save();
            }
        }
    }

    protected function handleException($e) {
        if($e instanceof Crossroads_API_ResponseException) {
            // Crossroads_API_ResponseExceptions should not be logged, they are responses
            // to be sent to the client
            return $this->getResponse()
                ->setHttpResponseCode($e->getStatusCode())
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([
                "message"   => $e->getMessage(),
                "errorCode" => $e->getCode(),
                "data"      => $e->getData()
            ]));
        }

        $this->getResponse()->setHttpResponseCode(500);

        Mage::logException($e);

        if(Mage::getIsDeveloperMode()) {
            $this->getResponse()
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([
                    "message"   => $e->getMessage(),
                    "errorCode" => null,
                    "data"      => [
                        "code"  => $e->getCode(),
                        "file"  => $e->getFile(),
                        "line"  => $e->getLine(),
                        "trace" => array_map(function($l) {
                            if(array_key_exists("class", $l)) {
                                return [
                                    "call" => $l["class"].$l["type"].$l["function"],
                                    "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                    "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                ];
                            }

                            return [
                                "call" => $l["function"],
                                "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                "line" => array_key_exists("line", $l) ? $l["line"] : null,
                            ];
                        }, $e->getTrace())
                    ]
                ]));
        }
    }
}
