<?php

class Crossroads_API_OrdersController extends Crossroads_API_Controller_Resource {
    /**
     * @api {get} /orders List orders
     * @apiName getAllCustomerOrders
     * @apiGroup Orders
     * @apiParam {integer} [limit=20] Maximum number of items per page, maximum 100
     * @apiParam {integer} [page=1] Page number
     */
    public function getAll() {
        $this->ensureLoggedIn();

        $pageSize   = min($this->getRequest()->getQuery("limit") ?: 20, 100);
        $page       = max($this->getRequest()->getQuery("page") ?: 1, 1);
        $helper     = Mage::helper("API/customer");
        $customer   = $helper->getCurrentCustomer();
        $addrHelper = Mage::helper("API/address");
        $orders     = Mage::getResourceModel('sales/order_collection')
            ->addFieldToSelect('*')
            ->addFieldToFilter('customer_id', $customer->getId())
            ->addFieldToFilter('state', ['in' => Mage::getSingleton('sales/order_config')->getVisibleOnFrontStates()])
            ->setOrder('created_at', 'desc');
        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $orders->getSelect()->limitPage($page, $pageSize);
        $orders->load();

        return [200, array_values(array_map(function($order) use ($helper, $addrHelper) {
            $quote = Mage::getModel("sales/quote")->load($order->getQuoteId());

            return [
                "id"              => $order->getIncrementId(),
                "status"          => $order->getStatus(),
                "paymentMethod"   => $order->getPayment()->getMethod(),
                "shippingMethod"  => $quote->getShippingAddress()->getShippingMethod(),
                "billingAddress"  => $addrHelper->prepareQuoteAddress($quote->getBillingAddress(), $quote),
                "shippingAddress" => $addrHelper->prepareQuoteAddress($quote->getShippingAddress(), $quote),
                "quote"           => Mage::helper("API/Cart")->formatQuote($quote),
                "email"           => $quote->getCustomerEmail() ?: null,
                "createdAt"       => gmdate("Y-m-d\TH:i:s\Z", strtotime($quote->getCreatedAt())),
            ];
        }, $orders->getItems()))];
    }

    /**
     * @api {get} /orders/:id Show order
     * @apiName getCustomerOrder
     * @apiGroup Orders
     */
    public function getItem($id) {
        $this->ensureLoggedIn();

        $order      = Mage::getModel("sales/order")->loadByIncrementId($id);
        $addrHelper = Mage::helper("API/address");
        $helper     = Mage::helper("API/customer");

        if(!$order || !$order->getIncrementId() || !$order->getPayment() || $order->getCustomerId() != $helper->getCurrentCustomer()->getId()) {
            return [404];
        }

        $quote = Mage::getModel("sales/quote")->load($order->getQuoteId());

        return [200, [
            "id"              => $order->getIncrementId(),
            "status"          => $order->getStatus(),
            "paymentMethod"   => $order->getPayment()->getMethod(),
            "shippingMethod"  => $quote->getShippingAddress()->getShippingMethod(),
            "billingAddress"  => $addrHelper->prepareQuoteAddress($quote->getBillingAddress(), $quote),
            "shippingAddress" => $addrHelper->prepareQuoteAddress($quote->getShippingAddress(), $quote),
            "quote"           => Mage::helper("API/Cart")->formatQuote($quote),
            "email"           => $quote->getCustomerEmail() ?: null,
            "createdAt"       => gmdate("Y-m-d\TH:i:s\Z", strtotime($quote->getCreatedAt())),
        ]];
    }
}
