<?php

class Crossroads_API_CartController extends Crossroads_API_Controller_Resource
{
    /**
     * @apiDefine cartItems
     *
     * @apiSuccess {Object[]}  items  The quote list
     * @apiSuccess {Integer} items.id
     * @apiSuccess {Object}  items.product  A lightweight product object
     * @apiSuccess {Integer} items.product.id
     * @apiSuccess {String}  items.product.sku
     * @apiSuccess {String}  items.product.name
     * @apiSuccess {String}  items.product.urlKey
     * @apiSuccess {Double}  items.product.price
     * @apiSuccess {Integer} [items.product.stockQty]  Amount in stock
     * @apiSuccess {String}  items.product.manufacturer
     * @apiSuccess {String}  items.product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}  items.qty
     * @apiSuccess {Double}  items.rowTotal  Total price for the row
     * @apiSuccess {Double}  items.rowTax
     * @apiSuccess {Object}  [items.attributes]  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer} items.attributes[attributeId] Attribute value
     * @apiSuccess {Array}   [items.options]
     * @apiSuccess {Integer} items.options.id
     * @apiSuccess {String}  items.options.title
     * @apiSuccess {Boolean} items.options.useAsDefault
     * @apiSuccess {Integer} items.options.position
     * @apiSuccess {Object}  items.options.value       The selected value for the complex product attribute
     * @apiSuccess {Integer} items.options.value.id
     * @apiSuccess {String}  items.options.value.label
     * @apiSuccess {Boolean} items.options.value.isPercent
     * @apiSuccess {Mixed}   [items.options.value.price]
     * @apiSuccess {String}  [items.options.value.thumbnail]  Thumbnail of the selected option
     * @apiSuccess {Object} summary  Cart summary
     * @apiSuccess {Double} summary.subTotal  Always includes tax
     * @apiSuccess {Double} summary.subTotalExclTax
     * @apiSuccess {Double} summary.grandTotal  Always includes tax
     * @apiSuccess {Double} summary.grandTotalExclTax
     * @apiSuccess {Double} [summary.discount]  Always includes tax
     * @apiSuccess {Double} summary.tax
     * @apiSuccess {Object[]} summary.taxRates
     * @apiSuccess {Double} summary.taxRates.percent
     * @apiSuccess {Double} summary.taxRates.amount
     * @apiSuccess {Double} summary.shippingAmount Note that this is affected by magento settings
     * @apiSuccess {Double} summary.shippingAmountExclTax
     * @apiSuccess {String} summary.quoteCurrencyCode
     * @apiSuccess {Double} summary.qty
     * @apiSuccess {Object}  [summary.coupon] Null if no coupon is applied
     * @apiSuccess {String}  summary.coupon.couponCode
     * @apiSuccess {String}  summary.coupon.ruleName
     * @apiSuccess {Boolean} summary.virtual If the quote only contains virtual items
     *
     * @apiSuccessExample {json} Success-Response:
     * { "items": [ { "id": 1071, "product": { "id": 17, "name": "Biobiljett SF Bio", "urlKey": "biobiljett-sf-bio", "price": 99, "stockQty": 500, "manufacturer": "Presentkort", "thumbnail": "http://example.com/media/thumbnail.jpg" }, "qty": 1, "rowTotal": 99, "rowTax": 5.6, "attributes": null, "options": null } ], "summary": { "subTotal": 99, "subTotalExclTax": 93.4, "grandTotal": 99, "grandTotalExclTax": 99, "tax": 5.6, "discount": null, "shippingAmount": 0, "shippingAmountExclTax": 0, "quoteCurrencyCode": "SEK", "qty": 1, "couponCode": null }, "virtual": true }
     */

    /**
     * @api {post} /cart Add product to cart
     * @apiHeader {String} Content-Type application/json
     * @apiName createCartItem
     * @apiGroup Cart
     *
     * @apiParam {Object} product           The product to add to cart
     * @apiParam {Integer} product.id       The product id
     * @apiParam {Number} qty               Number of items to add
     * @apiParam {Object}  [attributes]  Map of attribute id => attribute value for configurable products
     * @apiParam {Integer} attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function createItem()
    {
        $helper = Mage::helper("API/cart");
        $cart   = Mage::getSingleton('checkout/cart');

        $helper->updateItem($cart->getQuote(), $this->requestData());

        $cart->save();

        return $this->getAll();
    }

    protected function getCartRequestItemsData() {
        $data = $this->requestData();

        if( ! is_array($data) || ! array_key_exists("items", $data)) {
            throw Crossroads_API_ResponseException::create(400, "Request body must have an items array", null, 2011);
        }

        $stringKeys = array_filter(array_keys($data["items"]), 'is_string');

        if( ! empty($stringKeys)) {
            throw Crossroads_API_ResponseException::create(400, "Request body must have an items array", null, 2011);
        }

        return $data["items"];
    }

    /**
     * @api {put} /cart Replace cart
     * @apiHeader {String} Content-Type application/json
     * @apiDescription The body-structure matches the contents of the `items` property from `GET /cart`,
     *                 with optional properties for row-id and extra properties are ignored.
     *                 Unlike Update cart (`PATCH /cart`) this action will replace the contents and remove
     *                 any cart items which are not present in the request.
     * @apiName replaceCart
     * @apiGroup Cart
     *
     * @apiParam {Object[]} items                  List of cart items
     * @apiParam {Integer}  [items.id]             The cart item row id
     * @apiParam {Object}   items.product          The product to add to cart
     * @apiParam {Integer}  items.product.id       The product id
     * @apiParam {Number}   items.qty              Number of items to add
     * @apiParam {Object}   [items.attributes]     Map of attribute id => attribute value for configurable products
     * @apiParam {Integer}  items.attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function replaceAll()
    {
        $helper = Mage::helper("API/cart");
        $cart   = Mage::getSingleton('checkout/cart');

        $helper->setCartItems($cart->getQuote(), $this->getCartRequestItemsData());

        $cart->save();

        return $this->getAll();
    }

    /**
     * @api {patch} /cart Update cart
     * @apiHeader {String} Content-Type application/json
     * @apiDescription This is a version of the "Replace cart" (`PUT /cart`) action which will only add to
     *                 and change existing cart items, it will never remove cart items missing from the
     *                 request.
     * @apiName updateCart
     * @apiGroup Cart
     *
     * @apiParam {Object[]} items                  List of cart items
     * @apiParam {Integer}  [items.id]             The cart item row id
     * @apiParam {Object}   items.product          The product to add to cart
     * @apiParam {Integer}  items.product.id       The product id
     * @apiParam {Number}   items.qty              Number of items to add
     * @apiParam {Object}   [items.attributes]     Map of attribute id => attribute value for configurable products
     * @apiParam {Integer}  items.attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function updateAll() {
        $helper = Mage::helper("API/cart");
        $items  = $this->getCartRequestItemsData();
        $cart   = Mage::getSingleton('checkout/cart');

        // Multiple product requests
        foreach($items as $item) {
            $helper->updateItem($cart->getQuote(), $item);
        }

        $cart->save();

        return $this->getAll();
    }

    /**
     * @api {put} /cart/:id Update product cart item
     * @apiHeader {String} Content-Type application/json
     * @apiName replaceCartItem
     * @apiGroup Cart
     * @apiDescription Returns the contents of the whole cart despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number} id ID of cart item
     * @apiParam {Object} product
     * @apiParam {Integer} product.id
     * @apiParam {Number} qty New item quantity
     * @apiParam {Object}  [attributes]  Map of attribute id => attribute value for configurable products
     * @apiParam {Integer} attributes[attributeId] Attribute value
     *
     * @apiUse cartItems
     */
    public function replaceItem($id) {
        $params = $this->requestData();
        $helper = Mage::helper("API/cart");
        $cart   = Mage::getSingleton('checkout/cart');

        // Null-check required on $params
        if (empty($params) || empty($params["qty"])) {
            return [400];
        }

        $params["id"] = $id;

        $helper->updateItem($cart->getQuote(), $params);

        $cart->save();

        return $this->getAll();
    }

    /**
     * @api {delete} /cart/:id Remove product from cart
     * @apiName deleteCartItem
     * @apiGroup Cart
     * @apiDescription Returns the contents of the whole cart despite just replacing one item, reason is that
     * removing/modifying a product can alter more than just the row itself.
     *
     * @apiParam {Number} id  ID of cart item
     *
     * @apiUse cartItems
     */
    public function deleteItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $cart  = Mage::getSingleton("checkout/cart");
        $quote = $cart->getQuote();
        $item  = $quote->getItemById($id);

        if( ! $item) {
            return [404];
        }

        Mage::helper("API/cart")->removeItem($quote, $item);

        $cart->save();

        return $this->getAll();
    }

    /**
     * @api {get} /cart/:id Get product from cart
     * @apiName getCartItem
     * @apiGroup Cart
     *
     * @apiParam {Number} id  ID of cart item
     *
     * @apiSuccess {Integer} id
     * @apiSuccess {Object}  product  A lightweight product object
     * @apiSuccess {Integer} product.id
     * @apiSuccess {String}  product.sku
     * @apiSuccess {String}  product.name
     * @apiSuccess {String}  product.urlKey
     * @apiSuccess {Double}  product.price
     * @apiSuccess {mixed}   product.stockQty               Amount in stock, null if information not available
     * @apiSuccess {String}  product.manufacturer
     * @apiSuccess {String}  product.thumbnail  Complete URL to a thumbnail for the product
     * @apiSuccess {Double}  qty
     * @apiSuccess {Double}  rowTotal  Total price for the row
     * @apiSuccess {Object}  attributes  Map of attribute id => attribute value for configurable products
     * @apiSuccess {Integer} attributes[attributeId] Attribute value
     */
    public function getItem($id)
    {
        if (empty($id)) {
            return [404];
        }

        $cartItems = Mage::getModel('checkout/cart')->getQuote()->getAllItems();

        foreach($cartItems as $item) {
            if($item->getId() == $id) {
                return [200, Mage::helper('API/Product')->prepareCartProduct($item)];
            }
        }

        return [404];
    }

    /**
     * @api {get} /cart Get cart
     * @apiName getCartItems
     * @apiGroup Cart
     *
     * @apiUse cartItems
     */
    public function getAll()
    {
        $quote = Mage::getModel('checkout/session')->getQuote();

        $quote->collectTotals();

        return [200, Mage::helper("API/cart")->formatQuote($quote)];
    }

    /**
     * @api {delete} /cart Empty cart
     * @apiName emptyCart
     * @apiGroup Cart
     *
     * @apiSuccessExample {json} Empty-cart Response:
     * { "items": [], "summary": { "subTotal": 0, "subTotalExclTax": 0, "grandTotal": 0, "grandTotalExclTax": 0, "tax": null, "discount": null, "shippingAmount": 0, "quoteCurrencyCode": "SEK", "qty": null, "couponCode": null } }
     * @apiUse cartItems
     */
    public function deleteAll()
    {
        $cart  = Mage::getSingleton('checkout/cart');
        $items = $cart->getItems();

        foreach($items as $item) {
            $cart->removeItem($item->getItemId())->save();
        }

        return $this->getAll();
    }
}
