<?php

class Crossroads_API_Helper_Attributes {
    /**
     * Executed after an associated array has been prepared for the attribute-configuration,
     * this array can be modified.
     *
     * Params:
     *  * entity:        Entity
     *  * config:        Configuration array of frontend_input and attribute_code
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_POST_DATA_PREPARE = "crossroads_api_attributes_post_data_prepare";

    /**
     * Fetches the user-configured entity attributes for the supplied entity.
     *
     * @param  Mage_Catalog_Model_Product|Mage_Catalog_Model_Customer
     * @param  array   Array with attribute_code and frontend_input
     * @return object
     */
    public function getEntityAttributes($entity, $config)
    {
        $attributes = [];

        foreach (($config ?: []) as $attribute) {
            if(!is_array($attribute) ||
                !array_key_exists("attribute_code", $attribute) ||
                !array_key_exists("frontend_input", $attribute)) {
                continue;
            }

            $code  = $attribute["attribute_code"];
            $input = $attribute["frontend_input"];

            switch($input) {
                case "select":
                case "multiselect":
                    $attributes[$code] = $entity->getAttributeText($code) ?: $entity->getData($code) ?: null;
                    break;
                case "media_image":
                    $value       = $entity->getAttributeText($code) ?: $entity->getData($code) ?: null;
                    $mediaConfig = $entity->getMediaConfig();

                    $attributes[$code] = (empty($value) || $value === "no_selection") && $mediaConfig ? null :
                        $mediaConfig->getMediaUrl($value);
                    break;
                default:
                    $attributes[$code] = $entity->getData($code);
            }
        }

        $data = new Varien_Object($attributes);

        Mage::dispatchEvent(self::EVENT_POST_DATA_PREPARE, [
            "entity"        => $entity,
            "config"        => $config,
            "prepared_data" => $data,
        ]);

        return $data->getData() ?: new stdClass();
    }

    /**
     * Updates the custom attributes on the entity based on the configuration.
     *
     * @param  Mage_Customer_Model_Customer
     * @param  array    Array with attribute_code, frontend_input and editable.
     * @param  array    Input data
     * @return object
     */
    public function updateEntityAttributes($entity, $config, $newData) {
        foreach(($config ?: []) as $attribute) {
            if(!is_array($attribute) ||
               !array_key_exists("attribute_code", $attribute) ||
               !array_key_exists("editable", $attribute)) {
                continue;
            }

            if(strtolower(trim((string)$attribute["editable"])) === "true" && array_key_exists($attribute["attribute_code"], $newData)) {
                $entity->setData($attribute["attribute_code"], $newData[$attribute["attribute_code"]]);
            }
        }

        return $entity;
    }
}
