<?php

/**
 * Class managing payment methods.
 */
class Crossroads_API_Payment_Registry {
    protected $models = [];

    /**
     * Lists all registered payment methods.
     *
     * @return Array<Crossroads_API_Payment_Abstract>
     */
    public function getAll() {
        return $this->models;
    }

    /**
     * Returns a list of all registered payment codes.
     *
     * @return Array<string>
     */
    public function getAllCodes() {
        return array_keys($this->models);
    }

    /**
     * Returns true if a payment method with the supplied payment code exists in the registry.
     *
     * @param  string
     * @return boolean
     */
    public function hasMethod($paymentCode) {
        return array_key_exists($paymentCode, $this->models);
    }

    /**
     * Returns the payment model associated with the supplied payment code, throws an exception
     * if the associated payment method does not exist.
     *
     * @param  string
     * @return Crossroads_API_Payment_Abstract
     */
    public function getMethod($paymentCode) {
        if( ! array_key_exists($paymentCode, $this->models)) {
            throw new Exception("Crossroads_API: Invalid payment method code '$paymentCode'.");
        }

        return $this->models[$paymentCode];
    }

    /**
     * Adds a payment method.
     *
     * @param  Crossroads_API_Payment_Abstract
     */
    public function addMethod(Crossroads_API_Payment_Abstract $paymentMethod) {
        $paymentCode = $paymentMethod->getCode();

        if(array_key_exists($paymentCode, $this->models)) {
            throw new Exception("Crossroads_API: Duplicate payment method code '$paymentCode'.");
        }

        $this->models[$paymentCode] = $paymentMethod;
    }
}
