<?php

class Crossroads_API_Controller_Super extends Mage_Core_Controller_Front_Action
{
    /**
     * For outputting a JSON response
     *
     * @param int $status, HTTP code
     * @param mixed $data, the response data
     */
    protected function sendData($data) {
        if(!is_array($data) || !$data[0]) {
            throw new Exception("Missing return array.");
        }

        assert(gettype($data[0]) === "integer", __METHOD__.": \$status must be of type integer");

        $this->getResponse()->setHttpResponseCode($data[0]);

        if(array_key_exists(1, $data)) {
            $this->getResponse()
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode($data[1]));
        }
    }

    /**
     * Get JSON request data as an assoc array, will output null if Content-Type
     * is not application/json.
     *
     * NOTE: Does not accept NULL as a json string.
     *
     * @return array|null attached data, null if not application/json
     * @throws Crossroads_API_ResponseException  If the content is application/json but cannot be parsed
     */
    protected function requestData() {
        $request      = $this->getRequest();
        $content_type = $request->getHeader("Content-Type");

        if(strtolower(trim($content_type)) === "application/json") {
            $data = json_decode($request->getRawBody(), true);

            if($data === null) {
                throw Crossroads_API_ResponseException::create(400, "Could not parse JSON data", null, 100);
            }

            return $data;
        }

        return null;
    }

    protected function ensureLoggedIn() {
        if( ! Mage::helper("API/customer")->isLoggedIn()) {
            throw Crossroads_API_ResponseException::create(403, "Customer is not logged in", null, 3000);
        }
    }

    protected function handleException($e) {
        if($e instanceof Crossroads_API_ResponseException) {
            // Crossroads_API_ResponseExceptions should not be logged, they are responses
            // to be sent to the client
            return $this->getResponse()
                ->setHttpResponseCode($e->getStatusCode())
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode([
                "message"   => $e->getMessage(),
                "errorCode" => $e->getCode(),
                "data"      => $e->getData()
            ]));
        }

        $this->getResponse()->setHttpResponseCode(500);

        Mage::logException($e);

        if(Mage::getIsDeveloperMode()) {
            $this->getResponse()
                ->setHeader('Content-Type', 'application/json', true)
                ->setBody(json_encode([
                    "message"   => $e->getMessage(),
                    "errorCode" => null,
                    "data"      => [
                        "code"  => $e->getCode(),
                        "file"  => $e->getFile(),
                        "line"  => $e->getLine(),
                        "trace" => array_map(function($l) {
                            if(array_key_exists("class", $l)) {
                                return [
                                    "call" => $l["class"].$l["type"].$l["function"],
                                    "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                    "line" => array_key_exists("line", $l) ? $l["line"] : null,
                                ];
                            }

                            return [
                                "call" => $l["function"],
                                "file" => array_key_exists("file", $l) ? $l["file"] : null,
                                "line" => array_key_exists("line", $l) ? $l["line"] : null,
                            ];
                        }, $e->getTrace())
                    ]
                ]));
        }
    }
}
