<?php

class Crossroads_API_Helper_Product {
    /**
     * Executed after the basic product associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_POST_DATA_PREPARE      = "crossroads_api_product_post_data_prepare";

    /**
     * Executed after the basic product-list associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_LIST_POST_DATA_PREPARE = "crossroads_api_product_list_post_data_prepare";

    /**
     * Executed after the basic product-cart associated array has been prepared. This event allows
     * modification of the returned product associated array.
     *
     * Params:
     *  * product        The product object
     *  * item           The cart item
     *  * prepared_data  The associated array as a varien object, note that data is in
     *                   camel-case, so use `setData` and `getData` to modify.
     */
    const EVENT_CART_POST_DATA_PREPARE = "crossroads_api_product_cart_post_data_prepare";

    /**
     * Image-size, smallet dimension, pixels.
     *
     * @var number
     */
    protected $smallImageSize;
    /**
     * Image-size, large dimension, pixels.
     *
     * @var number
     */
    protected $largeImageSize;
    /**
     * Image-size, thumbnail dimension, pixels.
     *
     * @var number
     */
    protected $thumbImageSize;
    /**
     * If to include stock quantity in product detail, cart and cart options.
     *
     * @var boolean
     */
    protected $getStockQty        = false;
    /**
     * If to include media gallery in product detail.
     *
     * @var boolean
     */
    protected $getMediaGallery    = false;
    /**
     * If to include related products in product detail.
     *
     * @var boolean
     */
    protected $getRelatedProducts = false;
    /**
     * "Constant" list of product-types which has options.
     *
     * @var array
     */
    protected $TYPES_WITH_OPTIONS = ["configurable", "bundle"];
    protected $productAttributesView = null;
    protected $productAttributesList = null;
    protected $optionAttributes = null;
    /**
     * Customer group id => customer group name map.
     *
     * @var array
     */
    protected $customerGroups = [];

    /**
     * Loads store-specific configuration for product details and listings.
     */
    public function __construct()
    {
        $store = Mage::app()->getStore();

        $this->largeImageSize        = $store->getConfig('API_section/images/large_image_size');
        $this->smallImageSize        = $store->getConfig('API_section/images/small_image_size');
        $this->thumbImageSize        = $store->getConfig('API_section/images/thumbnail_image_size');
        $this->getStockQty           = $store->getConfig('API_section/product_view/get_stock_qty');
        $this->getMediaGallery       = $store->getConfig('API_section/product_view/get_media_gallery');
        $this->getRelatedProducts    = $store->getConfig('API_section/product_view/get_related_products');
        $this->getGroupPrices        = $store->getConfig('API_section/product_view/get_group_prices');
        $this->getGroupPricesInList  = $store->getConfig('API_section/product_list/get_group_prices_in_list');
        $this->getOptionsInList      = $store->getConfig('API_section/product_list/get_options_in_list');
        $this->productAttributesView = unserialize($store->getConfig('API_section/attributes/product_view'));
        $this->productAttributesList = unserialize($store->getConfig('API_section/attributes/product_list'));
        $this->optionAttributes      = unserialize($store->getConfig('API_section/attributes/product_option'));

        if($this->getGroupPrices || $this->getGroupPricesInList) {
            $this->customerGroups = $this->loadCustomerGroupMap();
        }
    }

    protected function loadCustomerGroupMap() {
        return array_map(function($group) {
            return $group->getCustomerGroupCode();
        }, Mage::getModel("customer/group")->getCollection()->getItems());
    }

    /**
     * Prepares a product for viewing in a category-listing, prepared for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareListProduct($product)
    {
        $small_image = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->resize($this->smallImageSize);

        $msrp = $this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$product->getMinimalPrice() / $msrp)) * 100)
            : null;

        $productData = new Varien_Object([
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),
            "urlKey"           => $product->getUrlKey(),

            "price"            => (double)$product->getMinimalPrice(),
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            // We need to manually fetch the group-price attribute since they are not included for a list product
            "groupPrices"      => $this->getGroupPricesInList ? $this->prepareGroupPrices(Mage::getResourceSingleton("catalog/product_attribute_backend_groupprice")->loadPriceData($product->getId(), Mage::app()->getStore()->getId())) : null,
            "originalPrice"    => (double)$product->getPrice(),
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,

            "shortDescription" => $product->getShortDescription(),
            "smallImage"       => $small_image,
            "isSalable"        => $product->getIsSalable() > 0,
            "options"          => $this->getOptionsInList ? $this->prepareOptions($product, false) : null,
            "attributes"       => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesList)
        ]);

        Mage::dispatchEvent(self::EVENT_LIST_POST_DATA_PREPARE, [
            "produt"        => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Converts a Product into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product data
     */
    public function prepareProductDetail($product) {
        $price           = (double)$product->getFinalPrice();
        $small_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'small_image')
            ->resize($this->smallImageSize);
        $large_image     = (string)Mage::helper("catalog/image")
            ->init($product, 'image')
            ->resize($this->largeImageSize);
        $original_image = (string)Mage::helper("catalog/image")
            ->init($product, 'image');

        $msrp = $this->getMsrp($product);

        $discount_percent = $product->getShowDiscount() && $msrp
            ? floor((1 - ((double)$price / $msrp)) * 100)
            : null;

        $inStock = $product->getIsInStock() > 0;

        // when the configurable products children is out of stock the price
        // gets calculated to "0". To avoid loading each child we simply assume
        // the product is out of stock.
        if ($product->getTypeId() === "configurable" && !$price) {
            $inStock = false;
        }

        $productData = new Varien_Object([
            "id"               => (int) $product->getEntityId(),
            "type"             => $product->getTypeId(),
            "name"             => $product->getName(),
            "sku"              => $product->getSku(),

            "price"            => $price,
            "specialPrice"     => $product->hasSpecialPrice() ? ((double)$product->getSpecialPrice()) : null,
            // Included normally when a product is ->load()ed
            "groupPrices"      => $this->getGroupPrices ? $this->prepareGroupPrices($product->getData("group_price") ?: []) : null,
            "categoryIds"      => array_map(function($id) { return (int)$id; }, $product->getCategoryIds() ?: []),
            "originalPrice"    => (double)$product->getPrice(),
            "msrp"             => $msrp,
            "discountPercent"  => $discount_percent,
            "metaDescription"  => $product->getMetaDescription(),
            "shortDescription" => $product->getShortDescription(),
            "description"      => $product->getDescription(),
            "urlKey"           => $product->getUrlKey(),
            "smallImage"       => $small_image,
            "largeImage"       => $large_image,
            "originalImage"    => $original_image,
            "isInStock"        => $inStock,
            "isSalable"        => $product->getIsSalable() > 0,
            "stockQty"         => $this->getStockQty ?
                (double)Mage::getModel('cataloginventory/stock_item')
                    ->loadByProduct($product)
                    ->getQty() : null,
            "options"          => $this->prepareOptions($product, true),
            "relatedProducts"  => $this->getRelatedProducts ? $this->prepareRelatedProducts($product) : null,
            "mediaGallery"     => $this->getMediaGallery ? $this->prepareMediaGallery($product) : null,
            "attributes"       => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesView),
        ]);

        Mage::dispatchEvent(self::EVENT_POST_DATA_PREPARE, [
            "produt"        => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Converts a Cart item into an associative array ready for JSON-serialization.
     *
     * @return Array Associative array of product cart data
     */
    public function prepareCartProduct($item)
    {
        $product  = $item->getProduct();
        $stockQty = null;
        $rowTotal = $item->getRowTotal()
                  + $item->getTaxAmount()
                  + $item->getHiddenTaxAmount()
                  - $item->getDiscountAmount();

        $thumbnail = (string)Mage::helper("catalog/image")
            ->init($product, 'thumbnail')
            ->resize($this->thumbImageSize);

        $attributes = null;
        $options    = $product->getTypeInstance(true)->getOrderOptions($product);

        if( ! empty($options["info_buyRequest"]) &&
            is_array($options["info_buyRequest"]) &&
            array_key_exists("super_attribute", $options["info_buyRequest"]) &&
            is_array($options["info_buyRequest"]["super_attribute"])) {
            $attributes = [];

            foreach($options["info_buyRequest"]["super_attribute"] as $k => $v) {
                $attributes["$k"] = (int)$v;
            }
        }

        $productData = new Varien_Object([
            "id"        => (int)$item->getId(),
            "product"   => [
                "id"         => (int)$product->getId(),
                "sku"        => $product->getSku(),
                "name"       => $product->getName(),
                "urlKey"     => $product->getUrlPath(),
                "price"      => (double)$item->getPriceInclTax(),
                "isSalable"  => $product->getIsSalable() > 0,
                "stockQty"   => $this->getStockQty ?
                    (double)Mage::getModel('cataloginventory/stock_item')
                        // This is the way to obtain the associated simple product, in this case
                        // it is carrying the stock data
                        ->loadByProduct($item->getOptionByCode('simple_product') ? $item->getOptionByCode("simple_product")->getProduct() : $product)
                        ->getQty() : null,
                "thumbnail"  => $thumbnail,
                "attributes" => Mage::helper("API/attributes")->getEntityAttributes($product, $this->productAttributesView)
            ],
            "qty"        => (double)$item->getQty(),
            "rowTotal"   => (double)$rowTotal,
            "rowTax"     => (double)$item->getTaxAmount(),
            "attributes" => $attributes,
            "options"    => $this->prepareOptionsCart($product, $options)
        ]);

        Mage::dispatchEvent(self::EVENT_CART_POST_DATA_PREPARE, [
            "item"          => $item,
            "product"       => $product,
            "prepared_data" => $productData,
        ]);

        return $productData->getData();
    }

    /**
     * Prepares product options for complex products from cart data.
     */
    protected function prepareOptionsCart($product, $options) {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return null;
        }
        if(empty($options["info_buyRequest"]) || !is_array($options["info_buyRequest"]["super_attribute"])) {
            return null;
        }

        $helper     = $this;
        $attrs      = $options["info_buyRequest"]["super_attribute"];
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = $instance->getUsedProducts(null, $product);

        // Specific options with specific set values
        return array_map(function($option) use($attrs, $children, $helper) {
            $value = null;
            $child = null;

            // Attempt to find the specific child-product for the selected attribute value
            foreach($children as $c) {
                if((int)$c->getData($option["attribute_code"]) === (int)$attrs[$option["attribute_id"]]) {
                    $child = $c;

                    break;
                }
            }

            // Find the selected attribute-value
            foreach($option["values"] as $v) {
                if((int)$attrs[$option["attribute_id"]] === (int)$v["value_index"]) {
                    $value = $v;

                    break;
                }
            }

            if($value && ! $child) {
                throw new Exception(sprintf("Cart attribute %s: %s failed to obtain a child product.", $option["attribute_code"], $v["value_index"]));
            }

            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                // There might be a case where we cannot find the selected value
                "value"        => $value ? [
                    "id"         => (int)$value["value_index"],
                    "sku"        => $child->getSku(),
                    "label"      => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                    "isPercent"  => $value["is_percent"] > 0,
                    "isInStock"  => $child->isInStock() > 0,
                    "isSalable"  => $child->getIsSalable() > 0,
                    "stockQty"   => $helper->getStockQty ?
                        (double)Mage::getModel('cataloginventory/stock_item')
                                ->loadByProduct($child)
                                ->getQty() : null,
                    "price"      => (double)$child->getPrice(),
                    "msrp"       => $child->getMsrp(),
                    "attributes" => Mage::helper("API/attributes")->getEntityAttributes($child, $helper->optionAttributes)
                ] : null
            ];
        }, $rawOptions);
    }

    protected function prepareGroupPrices($groupPrices) {
        $customerGroups = $this->customerGroups;
        $websiteId      = Mage::app()->getStore()->getWebsiteId();

        return array_values(array_filter(array_map(function($price) use($customerGroups, $websiteId) {
            if($price["website_id"] != $websiteId && $price["website_id"] != 0) {
                return null;
            }

            return [
                "groupCode" => array_key_exists($price["cust_group"], $customerGroups) ? $customerGroups[$price["cust_group"]] : null,
                "price"     => (double)$price["price"]
            ];
        }, $groupPrices)));
    }

    /**
     * Obtains the MSRP for a given product, if it is a complex product will will find the smallest MSRP
     * of the child products.
     *
     * @param  Mage_Catalog_Model_Product
     * @return double
     */
    protected function getMsrp(Mage_Catalog_Model_Product $product) {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return (double)$product->getMsrp();
        }

        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        $msrps = [];

        foreach($rawOptions as $option) {
            foreach($children as $child) {
                foreach ($option["values"] as $value)  {
                    if ($child->getData($option["attribute_code"]) === $value["value_index"]) {
                        $msrp = $child->getMsrp();
                        if ($msrp) {
                            $msrps[] = $msrp;
                        }
                    }
                }
            }
        }

        return !empty($msrps) ? (double)min($msrps) : null;
    }

    /**
     * Prepares product option-list for a complex product.
     *
     * @param  Mage_Catalog_Model_Product
     * @param  boolean     If to include stock for each variant
     * @return array|null  Null if it is a simple product
     */
    protected function prepareOptions(Mage_Catalog_Model_Product $product, $includeStock)
    {
        if (!in_array($product->getTypeId(), $this->TYPES_WITH_OPTIONS)) {
            return null;
        }

        $helper     = $this;
        $instance   = $product->getTypeInstance(true);
        $rawOptions = $instance->getConfigurableAttributesAsArray($product);
        $children   = array_filter($instance->getUsedProducts(null, $product), function($p) {
            return $p->isSaleable();
        });

        return array_map(function($option) use($children, $helper, $includeStock) {
            return [
                "id"           => (int)$option["attribute_id"],
                "code"         => $option["attribute_code"],
                "title"        => $option["store_label"] ?: $option["frontend_label"] ?: $option["label"],
                "useAsDefault" => $option["use_default"] > 0,
                "position"     => (int)$option["position"],
                "values"       => array_values(array_filter(array_map(function($value) use($children, $option, $helper, $includeStock) {
                    // Find the correct child product and output its data
                    foreach($children as $child) {
                        if($child->getData($option["attribute_code"]) !== $value["value_index"]) {
                            continue;
                        }

                        $small_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'small_image')
                            ->resize($this->smallImageSize);
                        $large_image     = (string)Mage::helper("catalog/image")
                            ->init($child, 'image')
                            ->resize($this->largeImageSize);
                        $original_image = (string)Mage::helper("catalog/image")
                            ->init($child, 'image');

                        // TODO: Maybe skip this in lists anyway?
                        if($this->getGroupPrices) {
                            $attribute = $child->getResource()->getAttribute("group_price");

                            if($attribute) {
                                $attribute->getBackend()->afterLoad($child);
                            }
                        }

                        return [
                            "id"            => (int)$value["value_index"],
                            "sku"           => $child->getSku(),
                            "smallImage"    => $small_image,
                            "largeImage"    => $large_image,
                            "originalImage" => $original_image,
                            "label"         => $value["store_label"] ?: $value["label"] ?: $value["default_label"],
                            "isPercent"     => $value["is_percent"] > 0,
                            "isInStock"     => $child->isInStock() > 0,
                            "isSalable"     => $child->getIsSalable() > 0,
                            "stockQty"      => $includeStock && $helper->getStockQty ?
                                (double)Mage::getModel('cataloginventory/stock_item')
                                    ->loadByProduct($child)
                                    ->getQty() : null,
                            "groupPrices"   => $this->getGroupPrices ? $this->prepareGroupPrices($child->getData("group_price") ?: []) : null,
                            "price"         => (double)$child->getPrice(),
                            "msrp"          => (double)$child->getMsrp(),
                            "attributes"    => Mage::helper("API/attributes")->getEntityAttributes($child, $helper->optionAttributes)
                        ];
                    }

                    // Empty, array_filter above will remove empty entries
                    return null;
                }, $option["values"])))
            ];
        }, $rawOptions);
    }

    protected function prepareMediaGallery($product)
    {
        $coll = $product->getMediaGalleryImages();

        if( ! $coll) {
            return [];
        }

        return array_values(array_map(function($image) use ($product) {
            return [
                "thumbnail" => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->resize($this->thumbImageSize),
                "image"     => (string)Mage::helper("catalog/image")
                    ->init($product, 'image', $image->getFile())
                    ->resize($this->smallImageSize),
            ];
        }, $coll->getItems()));
    }

    protected function prepareRelatedProducts($product)
    {
        return array_values(array_map([$this, "prepareListProduct"], $product->getRelatedProductCollection()
            ->addAttributeToSelect('*')
            ->setPositionOrder()
            ->addMinimalPrice()
            ->addStoreFilter(Mage::app()->getStore()->getStoreId())
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            )->getItems()));
    }
}
