<?php

/**
 * Class Crossroads_API_TestController
 *
 * This is locked to the Crossroads office IP
 */
class Crossroads_API_TestController extends Crossroads_API_Controller_Super
{
    const ALLOWED_IP = "5.150.194.196"; // crossroads firewall

    protected function isAllowed($s, $allowed) {
        $ip = !empty($s['HTTP_X_FORWARDED_FOR']) ? $s['HTTP_X_FORWARDED_FOR'] : $s['REMOTE_ADDR'];

        if ($ip === $allowed || Mage::getIsDeveloperMode()) {
            return true;
        }
        return false;
    }

    public function _construct()
    {
        if (!$this->isAllowed($_SERVER, self::ALLOWED_IP)) {
            header("HTTP/1.0 404 Not Found");
            exit();
        }
    }

    /**
     * @api {get} /test/mail Test email template
     * @apiName testEmailTemplate
     * @apiGroup Test
     *
     * @apiParam {String} template The id of the template to test
     * @apiParam {String} order The id of the order (optional)
     * @apiParam {String} variables, separated by commas (optional)
     */
    public function emailAction()
    {
        $template_id = $this->getRequest()->getParam("template");
        $order_id    = $this->getRequest()->getParam("order");
        $variables   = $this->getRequest()->getParam("variables");

        if (empty($template_id)) {
            return $this->sendData([400]);
        }

        $emailTemplate = Mage::getModel('core/email_template')->load($template_id);

        $emailTemplateVars = explode(",", $variables);

        if (!empty($order_id)) {
            $emailTemplateVars['order'] = Mage::getModel('sales/order')->load($order_id);

            $paymentBlock = Mage::helper('payment')->getInfoBlock($emailTemplateVars['order']->getPayment())->setIsSecureMode(true);
            $paymentBlock->getMethod()->setStore(Mage::app()->getStore());

            $emailTemplateVars['payment_html'] = $paymentBlock->toHtml();
        }

        echo $emailTemplate->getProcessedTemplate($emailTemplateVars);
    }

    public function quoteAction()
    {
        $quote = Mage::getSingleton('checkout/session')->getQuote();
        $this->sendData([200, $quote->getData()]);
    }

    /**
     * @api {get} /categories Get all categories
     * @apiName getCategories
     * @apiGroup Categories
     *
     * @apiSuccess {Object[]} categories Array of the categories
     * @apiSuccess {String} categories.id Category ID
     * @apiSuccess {String} categories.name Name of the category
     * @apiSuccess {String} categories.description Category description
     * @apiSuccess {String} categories.metaDescription
     * @apiSuccess {String} categories.urlKey
     * @apiSuccess {Integer} categories.position
     * @apiSuccess {Integer} categories.level
     * @apiSuccess {Boolean} categories.isAnchor
     * @apiSuccess {String} categories.path
     * @apiSuccess {Integer} categories.parentId
     * @apiSuccess {Boolean} categories.includeInMenu
     */
    public function getAll()
    {
        $store_id = Mage::app()->getStore()->getStoreId();
        $root_id  = Mage::app()->getStore($store_id)->getRootCategoryId();

        $categories = Mage::getModel('catalog/category')
            ->getCollection()
            ->setStoreId($store_id)
            ->addFieldToFilter('is_active', 1)
            ->addAttributeToFilter('path', ['like' => "1/{$root_id}/%"])
            ->addAttributeToSelect('*');
        $helper = Mage::helper("API/category");

        return [200, array_values(array_map(function($c) use($helper) {
            return $helper->prepareCategory($c);
        }, $categories->getItems()))];
    }

    /**
     * @api {get} /categories/:id Get category
     * @apiName getCategory
     * @apiGroup Categories
     *
     * @apiParam {Integer} id Category ID
     *
     * @apiSuccess {Object} category Category
     * @apiSuccess {String} category.id Category ID
     * @apiSuccess {String} category.name Name of the category
     * @apiSuccess {String} category.description Category description
     * @apiSuccess {String} category.metaDescription
     * @apiSuccess {String} category.urlKey
     * @apiSuccess {Integer} category.position
     * @apiSuccess {Array} category.images
     * @apiSuccess {Object} category.images.small
     * @apiSuccess {String} category.images.small.url
     * @apiSuccess {String} category.images.small.dataUrl
     * @apiSuccess {Object} category.images.medium
     * @apiSuccess {String} category.images.medium.url
     * @apiSuccess {String} category.images.medium.dataUrl
     * @apiSuccess {Object} category.images.large
     * @apiSuccess {String} category.images.large.url
     * @apiSuccess {String} category.images.large.dataUrl
     * @apiSuccess {Integer} category.level
     * @apiSuccess {Boolean} category.isAnchor
     * @apiSuccess {String} category.path
     * @apiSuccess {Integer} category.parentId
     * @apiSuccess {Boolean} category.includeInMenu
     *
     * @apiSuccess {Object} category.hero
     * @apiSuccess {String} category.hero.heading
     * @apiSuccess {String} category.hero.description
     * @apiSuccess {String} category.hero.offerIcon
     * @apiSuccess {Object} category.hero.button
     * @apiSuccess {String} category.hero.button.text
     * @apiSuccess {String} category.hero.button.link
     *
     * @apiSuccess {Object[]} products Array of the products in the category
     * @apiSuccess {Integer}  products.id Product id
     * @apiSuccess {String}   products.type Product type
     * @apiSuccess {String}   products.sku
     * @apiSuccess {String}   products.urlKey
     * @apiSuccess {Double}   products.price
     * @apiSuccess {Double}   products.specialPrice
     * @apiSuccess {Double}   products.msrp
     * @apiSuccess {Double}   products.discountPercent
     * @apiSuccess {String}   products.shortDescription
     * @apiSuccess {Mixed}    products.manufacturer String if a manufacturer is set, null otherwise
     * @apiSuccess {String}   products.smallImage
     * @apiSuccess {Boolean}  products.isSalable
     * @apiSuccess {Object[]} products.options
     */
    public function getItem($id)
    {
        $category = Mage::getModel('catalog/category')->load($id);

        if( ! $category || !$category->getId()) {
            return [404];
        }

        $collection = Mage::getResourceModel('catalog/product_collection')
            ->addAttributeToSelect('*')
            ->addCategoryFilter($category)
            ->addAttributeToFilter(
                'visibility',
                ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
            );

        $collection->load();

        $helper = Mage::helper("API/product");

        return [200, [
            "category" => Mage::helper("API/category")->prepareCategory($category),
            "products" => array_values(array_map(function($p) use($helper) {
                return $helper->prepareListProduct($p);
            }, $collection->getItems()))
        ]];
    }
}
