<?php

class Crossroads_API_RouteController extends Crossroads_API_Controller_Super
{
    const BASE_URL = "/api/route";

    public function indexAction() {
        return $this->sendData([404]);
    }

    // Using true here we tell Magento's router that we support all actions.
    public function hasAction($_action) { return true; }

    /**
     * @api {get} /route/:url Get dynamic resource by url
     * @apiName route
     * @apiGroup Route
     * @apiDescription Uses Magento's `url_rewrite` module to determine if a category
     *     or product resides at that URL. If so that category/product/redirect/CMS-page will be returned.
     *
     * @apiParam {String} url The slug to search the index by
     *
     * @apiSuccess {String} type Type of resource
     * @apiSuccess {Object} product Product object
     * @apiSuccess {Object} cmspage CMS page object
     * @apiSuccess {Object} category Category object
     * @apiSuccess {Mixed} redirect Redirect URL
     */
    public function norouteAction($_core_route = null)
    {
        if($this->getRequest()->getMethod() !== "GET") {
            return $this->sendData([405]);
        }

        $url = $this->getRequest()->getRequestUri();

        if(strpos($url, self::BASE_URL) !== 0) {
            return $this->sendData([404]);
        }

        $url = trim(substr($url, strlen(self::BASE_URL)), "/");

        $rewrite = Mage::getModel("core/url_rewrite")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->loadByRequestPath($url);

        $rewriteData = $rewrite->getData();

        // is redirect
        if (!empty($rewriteData["request_path"]) && !empty($rewriteData["target_path"]) &&
            !$rewriteData["category_id"] && !$rewriteData["product_id"]) {

            return $this->sendData([200, [
                "type"     => "redirect",
                "product"  => null,
                "cmsPage"  => null,
                "category" => null,
                "redirect" => $rewriteData["target_path"]
            ]]);
        }

        // is product
        if (!empty($rewriteData["product_id"])) {
            $product = Mage::getModel('catalog/product')->load($rewriteData["product_id"]);

            if( ! $product || !$product->getId()) {
                return $this->sendData([404]);
            }

            return $this->sendData([200, [
                "type"     => "product",
                "product"  => Mage::helper("API/product")->prepareProductDetail($product),
                "cmsPage"  => null,
                "category" => null,
                "redirect" => null
            ]]);
        }

        // is category
        if (!empty($rewriteData["category_id"])) {
            $category = Mage::getModel("catalog/category")->load($rewriteData["category_id"]);

            if (!$category || !$category->getId()) {
                return $this->sendData([404]);
            }

            $collection = Mage::getModel('catalog/product')
                ->getCollection()
                ->addCategoryFilter($category)
                ->addAttributeToSelect("*")
                ->addMinimalPrice()
                ->addAttributeToFilter(
                    'visibility',
                    ['neq' => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE]
                )
                ->load();

            $pHelper = Mage::helper("API/product");

            return $this->sendData([200, [
                "type"     => "category",
                "product"  => null,
                "cmsPage"  => null,
                "category" => [
                    "category" => Mage::helper("API/category")->prepareCategory($category, true),
                    "products" => array_values(array_map(function($p) use($pHelper) {
                        return $pHelper->prepareListProduct($p);
                    }, $collection->getItems()))
                ],
                "redirect" => null
            ]]);
        }

        // is cms page
        $id = Mage::helper('API/cms')->getCategoryIdByUrlKey($url);

        if (!empty($id)) {
            $page = Mage::helper('API/cms')->getItem($id);

            if ($page) {
                return $this->sendData([200, [
                    "type"     => "cmspage",
                    "product"  => null,
                    "cmsPage"  => $page,
                    "category" => null
                ]]);
            }
        }

        return $this->sendData([404]);
    }
}
